from django.contrib import admin
from django.db.models.aggregates import Count
from django.utils.encoding import force_text
from django.utils.translation import ugettext_lazy as _


class LessonProblemLinkGroupFilter(admin.SimpleListFilter):
    """
    Фильтр по группам моделей занятие-вопрос (на основе поля group
    модели LessonProblemLink)
    """
    all_value = '_all'
    without_group_value = '_null'

    title = _('Группы')
    parameter_name = 'group'

    def lookups(self, request, model_admin):
        """
        Формирование пунктов фильтра по группам
        """
        lesson_id = request.GET.get('lesson')
        items = []

        groups = (model_admin.model.objects.filter(lesson=lesson_id)
                  .values('group')
                  .order_by('group')
                  .annotate(Count('id')))

        for item in groups:
            if item['group']:
                text = 'Группа {group_num} ({count})'.format(
                    group_num=item['group'],
                    count=item['id__count'],
                )
                value = item['group']

            else:
                text = 'Без группы ({count})'.format(count=item['id__count'])
                value = self.without_group_value

            items.append((value, text))

        return items

    def queryset(self, request, queryset):
        """
        Фильтрация по конкретной группе либо по объектам без группы либо вывод
        всех объектов связей занятие-вопрос
        """
        if self.value() == self.without_group_value:
            return queryset.filter(group__isnull=True)

        if self.value() == self.all_value:
            return queryset.all()

        return queryset.filter(group=self.value())

    def default_value(self):
        """
        Значение параметра querystring "Все объекты"
        """
        return self.all_value

    def choices(self, cl):
        """
        Учитывает кастомный параметр "Все объекты", потому что при отсутствии
        параметра номера группы выводятся объекты без группы, а не все объекты
        """
        yield {
            'selected': self.value() == self.all_value,
            'query_string': cl.get_query_string(
                {self.parameter_name: self.all_value}, []),
            'display': _('All'),
        }
        for lookup, title in self.lookup_choices:
            yield {
                'selected': self.value() == force_text(lookup),
                'query_string': cl.get_query_string({
                    self.parameter_name: lookup,
                }, []),
                'display': title,
            }


class LessonTypeFilter(admin.SimpleListFilter):
    """
    Фильтр по видам владельцев и курсов
    """
    title = 'Типы занятий'

    parameter_name = 'course_type'
    LOOKUPS = (
        ('1', _('Оригинальные курсы')),
        ('2', _('Бесплатные курсы')),
        ('3', _('Курсы учителей')),
        ('4', _('Курсы контент-менеджеров')),
        ('5', _('Занятия учителей')),
        ('6', _('Занятия контент-менеджеров')),
        ('7', _('Занятия вне курсов')),
    )

    def lookups(self, request, model_admin):
        """
        Список возможных фильтров
        """
        return self.LOOKUPS

    def queryset(self, request, queryset):
        """
        Фильтрация значений по выбранному фильтру
        """
        if self.value() == self.LOOKUPS[0][0]:
            return queryset.filter(
                courselessonlink__course__copy_of__isnull=True)
        elif self.value() == self.LOOKUPS[1][0]:
            return queryset.filter(courselessonlink__course__free=True)
        elif self.value() == self.LOOKUPS[2][0]:
            return queryset.filter(
                courselessonlink__course__owner__is_teacher=True,
                courselessonlink__course__owner__is_content_manager=False,
            )
        elif self.value() == self.LOOKUPS[3][0]:
            return queryset.filter(
                courselessonlink__course__owner__is_content_manager=True)
        elif self.value() == self.LOOKUPS[4][0]:
            return queryset.filter(
                owner__is_teacher=True,
                owner__is_content_manager=False,
            )
        elif self.value() == self.LOOKUPS[5][0]:
            return queryset.filter(owner__is_content_manager=True)
        elif self.value() == self.LOOKUPS[6][0]:
            return queryset.filter(courselessonlink__isnull=True)
