from builtins import object

from dal import autocomplete

from django import forms
from django.core.exceptions import ValidationError
from django.utils.translation import ugettext_lazy as _

from kelvin.common.admin import InlineJSONFormField
from kelvin.common.form_mixins import InfoFormMixin
from kelvin.resources.fields import SlimModelMultipleChoiceField
from kelvin.resources.models import Resource

from .models import Lesson, LessonProblemLink
from .validators import LessonProblemLinkValidator


class MethodologyFormMixin(object):
    methodology = SlimModelMultipleChoiceField(
        required=False,
        queryset=Resource.objects.all(),
        widget=autocomplete.ModelSelect2Multiple(
            url='admin:resources_resource_autocomplete',
            attrs={
                'data-width': 800,
                'data-allow-clear': 'false',
                'data-placeholder': 'Введите идентификатор ресурса',
            },
        ),
    )


class ProblemLinkForm(forms.ModelForm):
    """
    Форма для связи занятие-вопрос
    """
    options = InlineJSONFormField(required=False, label=_('Свойства вопроса'))

    def clean(self):
        """
        Проверяет формат поля `options`, а также что для связи типа "Теория"
        указана теория, но не задача, и наоборот
        """
        cleaned_data = super(ProblemLinkForm, self).clean()

        if self.errors.get('options'):
            # если была обнаружена ошибка в поле, ничего не делаем
            return cleaned_data
        try:
            LessonProblemLinkValidator.validate_options_json(
                cleaned_data.get('options'),
                cleaned_data.get('type'),
            )
        except ValidationError as e:
            raise ValidationError({'options': e.messages})

        if (
            cleaned_data.get('type') == self.Meta.model.TYPE_THEORY and
            cleaned_data.get('problem')
        ):
            raise ValidationError('У связи-теории должна быть связь с '
                                  'ресурсом, но не с задачей')

        if (
            cleaned_data.get('type') != self.Meta.model.TYPE_THEORY and
            cleaned_data.get('theory')
        ):
            raise ValidationError('У связи с задачей или тренажером не должно стоять теории')

        return cleaned_data

    class Meta(object):
        model = LessonProblemLink
        fields = '__all__'


class LessonAdminForm(InfoFormMixin, forms.ModelForm):
    """
    Форма для админки занятия
    """
    class Meta(object):
        model = Lesson
        fields = (
            'owner',
            'name',
            'subject',
            'info',
            'created_by',
            'modified_by',
            'available_for_support',
        )


class CorpLessonAdminForm(
    InfoFormMixin,
    # MethodologyFormMixin,
    forms.ModelForm,
):
    class Meta(object):
        model = Lesson
        fields = (
            'owner',
            'name',
            'subject',
            'theme',
            # 'methodology',
            'info',
            'hooks',
            'created_by',
            'modified_by',
            'available_for_support',
        )
