from builtins import object, str

from model_utils import FieldTracker
from sortedm2m.fields import SortedManyToManyField

from django.conf import settings
from django.db import models
from django.template import Context, Template
from django.utils import timezone
from django.utils.encoding import python_2_unicode_compatible
from django.utils.functional import cached_property
from django.utils.translation import ugettext_lazy as _

from kelvin.common.fields import JSONField
from kelvin.common.model_mixins import AvailableForSupportMixin, InfoMixin, TimeStampMixin, UserBlameMixin
from kelvin.problems.answers import Answer as BaseAnswer
from kelvin.problems.answers import check_answer
from kelvin.problems.models import Problem, TextResource
from kelvin.resources.models import Resource
from kelvin.subjects.models import Subject, Theme

HOMEWORK_TEMPLATE_NAME = _('Домашняя работа от {0}')
HOMEWORK_DATE_FORMAT = '%d.%m.%Y'


@python_2_unicode_compatible
class TextTemplate(models.Model):
    """
    Шаблон для текстов
    """
    name = models.CharField(
        verbose_name=_('Название шаблона'),
        max_length=255,
    )
    template = models.TextField(verbose_name=_('Шаблон'))

    def __str__(self):
        return self.name

    def render(self, context_dict):
        """
        Рендеринг шаблона стандартными джанговыми средствами
        """
        template = Template(self.template)
        context = Context(context_dict)

        return template.render(context).strip()

    class Meta(object):
        verbose_name = _('Шаблон текста')
        verbose_name_plural = _('Шаблоны текстов')


@python_2_unicode_compatible
class BaseLesson(InfoMixin, TimeStampMixin):
    problems = models.ManyToManyField(
        Problem,
        through='lessons.LessonProblemLink',
        verbose_name=_('Задачи в занятии'),
        blank=True,
    )
    owner = models.ForeignKey(
        settings.AUTH_USER_MODEL,
        verbose_name=_('Автор занятия'),
    )
    name = models.CharField(
        verbose_name=_('Название учебного занятия'),
        max_length=255,
        default='',
    )
    subject = models.ForeignKey(
        Subject,
        verbose_name=_('Учебный предмет задачи'),
        blank=True,
        null=True,
    )

    Answer = BaseAnswer

    def get_points(self, user_answers):
        """
        Подсчет баллов за занятие по проверенным ответам пользователя
        Балл за занятие — сумма всех баллов за каждую задачу.

        :param user_answers:  словарь проверенных ответов пользователя
                              на вопросы занятия
        :return: количество баллов за ответы
        :rtype: int
        """
        # TODO EDU-274
        # в `user_answers` проставляем
        points = 0
        for problem_link in list(self.problem_links_dict.values()):
            answers = user_answers.get(str(problem_link.id))
            if problem_link.problem.custom_answer:
                problem_points = (
                    self.Answer.get_points_for_custom_answer(answers) or 0)
            else:
                problem_points = self.Answer.get_points(
                    answers,
                    problem_link.options['max_attempts'],
                    (problem_link.options.get('max_points') or
                     problem_link.problem.max_points),
                    (problem_link.options.get('count_type') or
                     self.Answer.ATTEMPTS_COUNT_TYPE),
                )
            if points is not None and problem_points is not None:
                points += problem_points

        return points

    def get_max_points(self, assigned=None):
        """
        Возвращает максимально возможный балл за решение урока
        # TODO возможно можно хранить посчитанным в уроке

        :param assigned: список идентификаторов назначенных задач, по которому
                         надо считать максимум баллов, или `None`, если
                         назначены все задачи
        """
        links = self.problem_links_dict
        if assigned is None:
            return sum(
                link.options.get('max_points') or link.problem.max_points
                for link in list(links.values())
            )
        return sum(link.options.get('max_points') or link.problem.max_points
                   for link in list(links.values())
                   if link.id in assigned)

    @staticmethod
    def check_answers(problems_dict, user_answers):
        """
        Проверка ответов пользователя на занятие

        :param problems_dict: словарь с вопросами
        :type problems_dict: dict {str: [Problem]}
        :param user_answers: словарь с ответами пользователя на вопросы занятия
        :return: проверенные ответы
        :rtype: dict {str: Answer}
        """
        checked_answers = {}
        for problem_link_id, answer_attempts in list(user_answers.items()):
            if problem_link_id in problems_dict:
                checked_answers[problem_link_id] = [
                    check_answer(problems_dict[problem_link_id], user_answer)
                    for user_answer in answer_attempts
                ]
        return checked_answers

    @staticmethod
    def get_homework_name():
        """
        Название домашнего задания
        """
        return HOMEWORK_TEMPLATE_NAME.format(
            timezone.now().strftime(HOMEWORK_DATE_FORMAT))

    @property
    def primary_scenario(self):
        """
        Возвращает первый основной сценарий или `None`
        """
        if not hasattr(self, '_primary_scenario'):
            try:
                self._primary_scenario = LessonScenario.objects.filter(
                    lesson=self, primary=True)[:1][0]
            except IndexError:
                self._primary_scenario = None
        return self._primary_scenario

    def set_primary_scenario(self, value):
        """
        Специально сделан, чтобы нельзя было установить значение через
        `primary_scenario`
        """
        self._primary_scenario = value

    @cached_property
    def problem_links_dict(self):
        """
        Словарь с обычными задачами в занятии
        """
        return {
            str(link.id): link
            for link in self.lessonproblemlink_set
                            .exclude(problem__isnull=True)
                            .select_related('problem')
        }

    def __str__(self):
        """
        Идентификатор занятия
        """
        return 'Lesson {0}'.format(self.pk)

    class Meta(object):
        abstract = True
        verbose_name = _('Учебное занятие')
        verbose_name_plural = _('Учебные занятия')


class Lesson(BaseLesson, AvailableForSupportMixin, UserBlameMixin):
    ALLOWED_HOOKS = ['start', 'stop', ]  # TODO: move it to env

    cover = models.ForeignKey(
        Resource,
        verbose_name=_('Обложка'),
        null=True,
        blank=True,
        related_name='cover_for_lessons',
    )
    theme = models.ForeignKey(
        Theme,
        verbose_name=_('Тема'),
        blank=True,
        null=True,
    )
    methodology = SortedManyToManyField(
        Resource,
        verbose_name=_('Файлы методики'),
        blank=True,
    )

    tracker = FieldTracker(fields=[
        'available_for_support',
    ])

    # hooks example:
    """
    {
        "start": {
          "method": "get|post|put|patch",
          "url": "<url value (string)">
          "auth_info": {
             "type": "tvm",
             "tvm_client_id": "<tvm client id",
          }
        },
        ...
    }
    """
    hooks = JSONField(
        verbose_name=_('Хуки на состояния модуля'),
        blank=True,
        default={},
        help_text=_('Настройки хуков для интеграции с внешними системами (например Я.Лицей)')
    )

    Answer = BaseAnswer

    @cached_property
    def theory_links_dict(self):
        """
        Словарь с обычными задачами в занятии
        """
        return {
            str(link.id): link
            for link in self.lessonproblemlink_set
                            .exclude(theory__isnull=True)
                            .select_related('theory')
        }

    # TODO данный метод был ошибочно отключен - стоит проверить логику, прежде чем включать
    # def clean(self):
    #     for hook in self.hooks.items():
    #         if hook[0] not in set(self.ALLOWED_HOOKS):
    #             raise ValidationError("Hook '{}' is not allowed or supported".format(hook[0]))



class LessonProblemLink(AvailableForSupportMixin, UserBlameMixin, models.Model):
    """m2m-модель занятие-вопрос"""
    TYPE_COMMON = 1
    TYPE_THEORY = 3
    TYPES = (
        (TYPE_COMMON, _('Обычная задача')),
        (TYPE_THEORY, _('Теория')),
    )

    DEFAULT_MAX_ATTEMPTS = 1000
    DEFAULT_SHOW_TIPS = True
    DEFAULT_LIVES = 5
    DEFAULT_TIME_LIMIT = 420  # 7 минут

    lesson = models.ForeignKey(
        Lesson,
        verbose_name=_('Занятие'),
    )
    type = models.IntegerField(
        verbose_name=_('Тип задачи'),
        choices=TYPES,
        default=TYPE_COMMON,
    )
    problem = models.ForeignKey(
        Problem,
        verbose_name=_('Вопрос'),
        blank=True,
        null=True,
    )
    theory = models.ForeignKey(
        TextResource,
        verbose_name=_('Теория'),
        blank=True,
        null=True,
    )
    order = models.PositiveIntegerField(
        verbose_name=_('Порядковый номер в занятии'),
    )
    options = JSONField(
        verbose_name=_('Свойства вопроса'),
        blank=True,
        null=True,
    )
    cm_order = models.CharField(
        verbose_name=_('Порядок задач у контент-менеджера'),
        max_length=32,
        blank=True,
        default='',
    )
    group = models.IntegerField(
        verbose_name=_('Группа вопросов для вариаций'),
        blank=True,
        null=True,
        default=None,
    )
    block_id = models.IntegerField(
        verbose_name=_('Номер блока'),
        blank=True,
        null=True,
        default=None,
    )

    start_date = models.DateTimeField(
        verbose_name=_('Дата начала'),
        blank=True,
        null=True,
        default=None,
    )

    finish_date = models.DateTimeField(
        verbose_name=_('Дата окончания'),
        blank=True,
        null=True,
        default=None,
    )

    tracker = FieldTracker(fields=[
        'available_for_support',
    ])

    def get_max_points(self):
        return self.options.get('max_points') or self.problem._calculate_max_points()

    def save(self, **kwargs):
        """
        Добавляет дефолтное значение параметров
        """
        if self.type == self.TYPE_COMMON:
            if self.options is None:
                self.options = {
                    'max_attempts': self.DEFAULT_MAX_ATTEMPTS,
                    'max_points': self.problem and self.problem.max_points,
                    'show_tips': self.DEFAULT_SHOW_TIPS,
                }
        return super(LessonProblemLink, self).save(**kwargs)

    class Meta(object):
        verbose_name = _('Сценарий вопроса в занятии')
        verbose_name_plural = _('Сценарии вопросов в занятии')
        ordering = ['order']


class LessonScenario(models.Model):
    """
    Поля, задающие сценарий прохождения занятия
    """
    TRAINING_MODE = 1
    CONTROL_WORK_MODE = 2
    WEBINAR_MODE = 3
    DIAGNOSTICS_MODE = 4

    LESSON_MODES = (
        (TRAINING_MODE, _('Тренировка')),
        (CONTROL_WORK_MODE, _('Контрольная работа')),
        (WEBINAR_MODE, _('Вебинар')),
        (DIAGNOSTICS_MODE, _('Диагностика')),
    )

    # режимы занятий, для которых актуально время публикации результатов
    EVALUATION_LESSON_MODES = (
        CONTROL_WORK_MODE,
        DIAGNOSTICS_MODE,
    )

    class VisualModes(object):
        SEPARATE = 0
        BLOCKS = 1
        CHOICES = (
            (SEPARATE, 'По отдельности'),
            (BLOCKS, 'Блоками'),
        )

    lesson = models.ForeignKey(
        'lessons.Lesson',
        verbose_name=_('Занятие'),
    )
    mode = models.IntegerField(
        verbose_name=_('Режим занятия'),
        choices=LESSON_MODES,
        default=TRAINING_MODE,
    )
    duration = models.IntegerField(
        verbose_name=_('Длительность занятия (минуты)'),
        blank=True,
        null=True,
    )
    finish_date = models.DateTimeField(
        verbose_name=_('Время окончания контрольной'),
        blank=True,
        null=True,
    )
    evaluation_date = models.DateTimeField(
        verbose_name=_('Время публикации результатов контрольной'),
        blank=True,
        null=True,
    )
    show_answers_in_last_attempt = models.BooleanField(
        verbose_name=_('Показывать верные ответы в последней попытке "группы" попыток'),
        default=True,
    )
    url = JSONField(
        verbose_name=_('Адреса для вебинара'),
        blank=True,
        null=True,
    )
    start_date = models.DateTimeField(
        verbose_name=_('Дата и время начала трансляции вебинара'),
        blank=True,
        null=True,
    )
    visual_mode = models.IntegerField(
        choices=VisualModes.CHOICES,
        verbose_name=_('Отображать задачи'),
        default=VisualModes.SEPARATE,
    )
    allow_time_limited_problems = models.BooleanField(
        verbose_name=_('Разрешены задачи, ограниченные по времени'),
        blank=False,
        null=False,
        default=False,
    )
    max_attempts_in_group = models.IntegerField(
        verbose_name=_('Количество возможных попыток "в группе"'),
        default=2,
    )
    show_all_problems = models.BooleanField(
        verbose_name=_('Показывать сразу все задачи'),
        default=True,
    )
    comment = models.TextField(
        verbose_name=_('Комментарий к вебинару'),
        blank=True,
    )
    diagnostics_text = models.ForeignKey(
        TextTemplate,
        verbose_name=_('Шаблон текста для результатов диагностики'),
        blank=True,
        null=True,
    )
    primary = models.BooleanField(
        verbose_name=_('Основной сценарий'),
        default=False,
    )

    def clean(self):
        """
        Библиотечный метод, валидирует модель перед сохранением.
        Эта валидация работает при сохранении моделей в админке.
        """
        self.get_validator().validate(self)
        super(LessonScenario, self).clean()

    @staticmethod
    def get_validator():
        """
        Возвращает валидатор для данной модели
        """
        # FIXME Should not survive!
        from kelvin.lessons.validators import AbstractLessonScenarioValidator

        return AbstractLessonScenarioValidator

    @property
    def can_be_evaluated(self):
        """
        Сценарий, результаты которого могуть быть опубликованы
        """
        return self.mode in self.EVALUATION_LESSON_MODES

    class Meta(object):
        verbose_name = _('Сценарий занятия')
        verbose_name_plural = _('Сценарии занятий')
