from builtins import object
from collections import defaultdict

from rest_framework.serializers import JSONField

from kelvin.common.serializer_fields import NestedForeignKeyField
from kelvin.lessons.models import Lesson, LessonProblemLink
from kelvin.lessons.serializers import LessonInCourseSerializer, LessonShortSerializer
from kelvin.problems.serializers import AnswerDatabaseSerializer, AnswerSerializer


class ExpandableLessonField(NestedForeignKeyField):
    """
    Сериализатор вопроса, который сериализует полностью вопрос лишь при
    наличии параметра в контексте
    """
    class Meta(object):
        serializer = LessonInCourseSerializer
        model = Lesson

    def to_representation(self, value):
        """
        Смотрим, разворачивать или нет задачи
        """
        if self.context.get('expand_lessons'):
            return super(ExpandableLessonField, self).to_representation(value)
        else:
            return LessonShortSerializer().to_representation(value)


class AnswersField(JSONField):
    """
    Сериализация поля ответов в попытке
    """
    def to_representation(self, value):
        """
        Через сериализатор для базы создаем объекты вопросов,
        через сериализатор для клиента сериализуем
        """
        lesson_problem_links = {
            int(link.id): link for link in
            LessonProblemLink.objects.filter(pk__in=list(value.keys()))
            .select_related('problem')
        }
        ret = defaultdict(list)
        for lesson_problem_id, answers in list(value.items()):
            if int(lesson_problem_id) not in lesson_problem_links:
                # Check for nonexistent ids in value
                continue
            db_serializer = AnswerDatabaseSerializer(data=answers, many=True)
            if db_serializer.is_valid():
                self.context['lesson_problem_link'] = (
                    lesson_problem_links[int(lesson_problem_id)])
                ret[lesson_problem_id] = AnswerSerializer(
                    context=self.context,
                    many=True,
                ).to_representation(db_serializer.save())
        return super(AnswersField, self).to_representation(ret)
