from builtins import object

from django.db import transaction

from rest_framework import serializers

from kelvin.common.serializer_mixins import DateUpdatedFieldMixin, InfoFieldMixin

from ...resources.serializers import ResourceSerializerCorp
from ..models import Lesson
from . import LessonInCLessonSerializer  # isort: skip
from .lesson_problem_link import LessonProblemLinkSerializer
from .lesson_scenario import LessonScenarioSerializer


class LessonShortSerializer(DateUpdatedFieldMixin, serializers.ModelSerializer):
    """
    Сериализатор для краткого вида занятия
    """
    max_points = serializers.SerializerMethodField()

    def get_max_points(self, lesson):
        return sum(
            link.options.get('max_points') or link.problem.max_points
            for link in lesson.lessonproblemlink_set.all() if link.problem_id is not None
        )

    class Meta(object):
        model = Lesson
        fields = (
            'id',
            'date_updated',
            'max_points',
        )


class LessonSerializer(InfoFieldMixin, LessonInCLessonSerializer):
    """
        Сериализатор для поля `lessons` со сценарием
        """
    scenario = LessonScenarioSerializer(
        source='primary_scenario',
    )

    problems = LessonProblemLinkSerializer(
        source='lessonproblemlink_set',
        many=True,
    )
    methodology = serializers.ListSerializer(child=ResourceSerializerCorp(), read_only=True)

    class Meta(object):
        model = Lesson
        fields = (
            'id',
            'date_updated',
            'name',
            'owner',
            'theme',
            'scenario',
            'problems',
            'methodology',
            'info',
        )
        read_only_fields = (
            'owner',
            'date_updated',
            'methodology',
        )

    def save(self, **kwargs):
        """
        Сохраняем занятие, потом сохраняем сценарий
        """
        with transaction.atomic():
            serializer = self.fields['scenario']
            scenario_data = self.validated_data.pop(serializer.source, None)
            lesson = super(LessonSerializer, self).save(**kwargs)

            if scenario_data is not None:
                serializer._validated_data = scenario_data
                serializer.instance = lesson.primary_scenario
                serializer._errors = {}
                lesson.set_primary_scenario(serializer.save(lesson=lesson, primary=True))

            return lesson
