from builtins import object
from collections import OrderedDict

from rest_framework import serializers
from rest_framework.fields import SkipField
from rest_framework.relations import PKOnlyObject

from kelvin.common.serializer_mixins import DateUpdatedFieldMixin, ExcludeForStudentMixin, SerializerManyToManyMixin
from kelvin.resources.serializers import ResourceSerializerCorp

from ..models import Lesson
from .lesson_problem_link import BaseLessonProblemLinkSerializer, LessonProblemLinkSerializer


class BaseLessonInCLessonSerializer(
    DateUpdatedFieldMixin,
    SerializerManyToManyMixin,
    serializers.ModelSerializer
):
    """
        Сериализатор для курсозанятия без сценария
        """
    problems = BaseLessonProblemLinkSerializer(
        source='lessonproblemlink_set',
        many=True,
    )
    owner = serializers.PrimaryKeyRelatedField(
        required=False,
        read_only=True,
        default=serializers.CreateOnlyDefault(
            serializers.CurrentUserDefault()
        ),
    )
    subject = serializers.CharField(source='subject.slug', required=False,
                                    read_only=True)

    m2m_update_fields = {
        'problems': 'lesson',
    }
    m2m_skip_update = ('problems',)

    PREFETCH_LIST = (
        'theory',
        'theory__themes',
        'theory__resources',
        'theory__content_type_object',
        'theory__content_type_object__resource',
        'problem',
        'problem__subject',
        'problem__resources',
        'problem__meta',
        'problem__meta__main_theme',
        'problem__meta__additional_themes',
        'problem__meta__group_levels',
    )

    def to_representation(self, instance):
        """
        Переопределяю родительский метод и насильно добавляю prefetch-и,
        которые не доходят сюда из CLessonViewSet
        """
        ret = OrderedDict()
        fields = self._readable_fields

        for field in fields:
            try:
                attribute = field.get_attribute(instance)
            except SkipField:
                continue
            check_for_none = attribute.pk if \
                isinstance(attribute, PKOnlyObject) else attribute
            if check_for_none is None:
                ret[field.field_name] = None
            else:
                if field.field_name == 'problems':
                    attribute = attribute.prefetch_related(
                        *self.PREFETCH_LIST
                    )
                ret[field.field_name] = field.to_representation(attribute)

        return ret

    def validate(self, attrs):
        """Проставляем порядок в связях"""
        if 'lessonproblemlink_set' in attrs:
            for i, problem_link in enumerate(
                    attrs['lessonproblemlink_set'], 1):
                problem_link['order'] = i

        return super(BaseLessonInCLessonSerializer, self).validate(attrs)

    class Meta(object):
        model = Lesson
        fields = (
            'id',
            'date_updated',
            'name',
            'owner',
            'problems',
            'subject',
        )
        read_only_fields = (
            'owner',
            'date_updated',
            'subject',
        )


class CorpLessonInCLessonSerializer(
    ExcludeForStudentMixin,
    BaseLessonInCLessonSerializer
):
    """
    Сериализатор для курсозанятия без сценария
    """
    problems = LessonProblemLinkSerializer(
        source='lessonproblemlink_set',
        many=True,
    )
    methodology = serializers.ListSerializer(child=ResourceSerializerCorp(), read_only=True)

    PREFETCH_LIST = BaseLessonInCLessonSerializer.PREFETCH_LIST + (
        'problem__meta__skills',
        'problem__meta__problemmetaexam_set',
        'problem__meta__problemmetaexam_set__exam',
    )

    class Meta(object):
        model = Lesson
        fields = BaseLessonInCLessonSerializer.Meta.fields + (
            'theme',
            'methodology',
        )
        read_only_fields = (
            BaseLessonInCLessonSerializer.Meta.read_only_fields +
            ('methodology',)
        )
        exclude_for_student = {
            'methodology',
        }


class LessonInCLessonSerializer(CorpLessonInCLessonSerializer):
    problems = LessonProblemLinkSerializer(
        source='lessonproblemlink_set',
        many=True,
    )
    methodology = serializers.ListSerializer(child=ResourceSerializerCorp(), read_only=True)
