from builtins import object

from django.core.exceptions import ValidationError

from rest_framework import serializers

from kelvin.common.serializer_fields import PrefetchRelatedField, SelectRelatedField
from kelvin.common.serializers import ManyToManyListSerializer
from kelvin.problems.models import Problem, TextResource
from kelvin.problems.serializer_fields import ExpandableProblemField, ExpandableTextResourceField

from ..models import LessonProblemLink
from ..validators import LessonProblemLinkValidator


class BaseLessonProblemLinkSerializer(serializers.ModelSerializer):
    """
    Сериализатор связи занятие-вопрос
    """
    id = serializers.IntegerField(required=False)
    problem = ExpandableProblemField(allow_null=True, required=False)
    theory = ExpandableTextResourceField(allow_null=True, required=False)

    default_error_messages = {
        'type mismatch': u'Link type does not match its content',
    }

    prefetch_fk_fields = (
        PrefetchRelatedField(
            'problem',
            Problem.objects.all(),
        ),
        PrefetchRelatedField(
            'theory',
            TextResource.objects.all()
        ),
        SelectRelatedField('problem'),
        SelectRelatedField('theory'),
    )

    class Meta(object):
        model = LessonProblemLink
        fields = (
            'id',
            'type',
            'problem',
            'theory',
            'options',
            'block_id',
            'start_date',
            'finish_date',
        )
        list_serializer_class = ManyToManyListSerializer
        extra_kwargs = {
            # делаем тип обязательным полем, потому что при частичном
            # обновлении занятия, валидация вызывается без объекта связи
            # занятие-вопрос, и мы не можем валидировать
            'type': {'required': True},
        }

    def validate(self, attrs):
        """
        Проверяет поле `options`, а также то, что тип связи соответствует
        содержимому
        """
        try:
            # проверяем разметку вопросов
            LessonProblemLinkValidator.validate_options_json(
                attrs.get('options'), attrs.get('type'))
        except ValidationError as e:
            raise serializers.ValidationError({'options': e.messages})

        if (
            attrs.get('type') == self.Meta.model.TYPE_COMMON and
            attrs.get('theory') or
            attrs.get('type') == self.Meta.model.TYPE_THEORY and
            attrs.get('problem')
        ):
            self.fail('type mismatch')
        return attrs


class CorpLessonProblemLinkSerializer(BaseLessonProblemLinkSerializer):
    real_max_points = serializers.SerializerMethodField()

    class Meta(BaseLessonProblemLinkSerializer.Meta):
        model = LessonProblemLink
        fields = BaseLessonProblemLinkSerializer.Meta.fields + (
            'real_max_points',
        )

    def get_real_max_points(self, lesson):
        """
        Получаем действительный установленный max_points
        Приоритет у max_points в options problemLink'a
        :param lesson:
        :return:
        """
        if lesson.problem is None:
            return 0

        if lesson.options is None:
            return lesson.problem.max_points

        return (
            lesson.problem.max_points
            if lesson.options.get('max_points') is None else
            lesson.options.get('max_points')
        )


class LessonProblemLinkSerializer(CorpLessonProblemLinkSerializer):
    pass
