from builtins import zip
from typing import Iterable

from django.contrib.postgres.aggregates import BoolOr

from kelvin.lessons.models import Lesson, LessonProblemLink
from kelvin.problems.models import Problem


def copy_lesson(lesson, owner=None, include_problem_link_ids=None, name=None):
    """
    Копирует занятие и связи с задачами, изменяет объект текущего занятия

    :param owner: владелец копии
    :param include_problem_ids: идентификаторы задач данного занятия,
                                которые включить в копию
    :param name: название копии
    :return: словарь с новым идентификатором занятия и соответствием
             идентификатор старой связи - идентификатор новой связи
    """
    if include_problem_link_ids:
        problem_links = list(lesson.lessonproblemlink_set.filter(id__in=include_problem_link_ids))
    else:
        problem_links = list(lesson.lessonproblemlink_set.all())

    methodology = lesson.methodology.all()

    old_problem_link_ids = [link.id for link in problem_links]
    lesson.pk = None

    if owner:
        lesson.owner = owner

    if name:
        lesson.name = name

    lesson.edu_id = None
    lesson.save()

    new_lesson_id = lesson.pk

    # копируем связи с вопросами
    for problem_link in problem_links:
        problem_link.pk = None
        problem_link.lesson = lesson
    LessonProblemLink.objects.bulk_create(problem_links)

    # TODO предполагаем, что возвращаемый порядок будет таким же,
    # как при сохранении `bulk_create`
    new_problem_link_ids = LessonProblemLink.objects.filter(
        lesson_id=new_lesson_id,
    ).values_list('id', flat=True).order_by('id')

    # копируем связи с методологией
    lesson.methodology = methodology

    return {
        'new_lesson_id': new_lesson_id,
        'problems': dict(list(zip(old_problem_link_ids, new_problem_link_ids))),
    }


def update_available_for_support_on_lessons(lessons: Iterable[Lesson], available_for_support: bool) -> None:
    lesson_problem_link_to_update = LessonProblemLink.objects.filter(lesson__in=lessons)
    lesson_problem_link_to_update.update(available_for_support=available_for_support)

    if available_for_support:
        update_available_for_support_on_lesson_problem_links(lesson_problem_link_to_update)
    else:
        update_unavailable_for_support_on_lesson_problem_links(lesson_problem_link_to_update)


def update_available_for_support_on_lesson_problem_links(lesson_problem_links: Iterable[LessonProblemLink]) -> None:
    problems_to_update = Problem.objects.filter(
        id__in=LessonProblemLink.objects.filter(
            id__in=[lpl.id for lpl in lesson_problem_links],
        ).values_list('problem_id', flat=True),
    )
    problems_to_update.update(available_for_support=True)


def update_unavailable_for_support_on_lesson_problem_links(lesson_problem_links: Iterable[LessonProblemLink]) -> None:
    problems_to_update = Problem.objects.filter(
        id__in=LessonProblemLink.objects.filter(
            id__in=[lpl.id for lpl in lesson_problem_links],
        ).values_list('problem_id', flat=True),
    ).annotate(
        max_available_for_support=BoolOr('lessonproblemlink__available_for_support'),
    ).filter(
        max_available_for_support=False,
    )
    problems_to_update.update(available_for_support=False)


def update_available_for_support_on_lesson(lesson: Lesson) -> None:
    if lesson.tracker.has_changed('available_for_support'):
        update_available_for_support_on_lessons(lessons=[lesson], available_for_support=lesson.available_for_support)


def update_available_for_support_on_lesson_problem_link(lesson_problem_link: LessonProblemLink) -> None:
    if lesson_problem_link.problem_id is not None:
        if lesson_problem_link.tracker.has_changed('available_for_support'):
            update_available_for_support_on_lesson_problem_links([lesson_problem_link])
        else:
            update_unavailable_for_support_on_lesson_problem_links([lesson_problem_link])
