import logging

from celery.utils.log import get_task_logger

from django.contrib.postgres.functions import TransactionNow
from django.db import transaction
from django.db.models import F

from kelvin.celery import app
from kelvin.common.decorators.task_logger import logged_task
from kelvin.common.func_expressions import Interval
from kelvin.courses.models import CourseLessonLink
from kelvin.results.models import CourseLessonResult, CourseLessonResultHook
from kelvin.results.tasks import send_hook_task

logger = get_task_logger(__name__)
sentry_logger = logging.getLogger('sentry')


@logged_task
@app.task()
def process_staled_tests():
    """
    Находит все контрольные, такие что:
    - по контрольной прошел дедлайн
    - CourseLessonResult.completed = False
    - настроены хуки
    и для каждой такой контрольной дергает stop-хук
    """
    with transaction.atomic():
        uncompleted_results = CourseLessonResult.objects.select_for_update(skip_locked=True).filter(
            summary__clesson__mode=CourseLessonLink.CONTROL_WORK_MODE,
            date_created__lt=TransactionNow() - Interval('1 minute') * F("summary__clesson__duration"),
            completed=False,
            summary__clesson__lesson__hooks__has_key="stop",
        ).only("id", "summary__clesson__lesson__hooks")

        logger.info(f"Got {uncompleted_results.count()} staled tests with 'stop' hook")

        for item in uncompleted_results:
            result_id = item.id
            stop_hook_info = item.summary.clesson.lesson.hooks['stop']

            logger.debug("Creating hook '{}' of the course_lessonresult {}".format("stop", result_id))
            hook_obj, created = CourseLessonResultHook.objects.get_or_create(
                course_lesson_result_id=result_id,
                hook_info__has_key="stop",
                defaults={
                    "hook_info": {"stop": stop_hook_info},
                }
            )

        uncompleted_results.update(completed=True)  # Do not call save here !!!
