from builtins import object

import jsonschema

from django.core.exceptions import NON_FIELD_ERRORS, ValidationError
from django.utils.translation import ugettext_lazy as _

from rest_framework.settings import api_settings

from kelvin.problems.answers import Answer

from .models import LessonProblemLink, LessonScenario

COMMON_OPTIONS_SCHEMA = {
    'type': 'object',
    'properties': {
        'max_attempts': {
            'type': 'integer',
        },
        'show_tips': {
            'type': 'boolean',
        },
        'max_points': {
            'oneOf': [
                {
                    'type': 'integer',
                },
                {
                    'type': 'object',
                    'patternProperties': {
                        r'^\d+$': {
                            'type': 'integer',
                        },
                    },
                    'additionalProperties': False,
                },
            ]
        },
        'count_type': {
            'enum': [Answer.ATTEMPTS_COUNT_TYPE, Answer.MISTAKES_COUNT_TYPE],
        },
    },
    'required': ['max_attempts', 'show_tips'],
    'additionalProperties': False,
}

THEORY_OPTIONS_SCHEMA = {
    'type': 'object',
    'properties': {},
    'additionalProperties': False,
}


class LessonProblemLinkValidator(object):
    """
    Валидаторы связи занятие-вопрос
    """

    @staticmethod
    def validate_options_json(
            options,
            link_type=None,
    ):
        """
        Проверяет соответствие поля json-схеме
        """
        if link_type is None:
            link_type = LessonProblemLink.TYPE_COMMON
        try:
            if link_type == LessonProblemLink.TYPE_THEORY:
                if options is None:
                    return
                jsonschema.validate(options, THEORY_OPTIONS_SCHEMA)
            else:
                # не записываем в схему как `oneOf` для более понятных ошибок
                if options is None:
                    return
                jsonschema.validate(options, COMMON_OPTIONS_SCHEMA)
        except jsonschema.ValidationError as e:
            raise ValidationError(e.message)


class AbstractLessonScenarioValidator(object):

    lesson_modes = dict(LessonScenario.LESSON_MODES)

    required_fields = {
        LessonScenario.CONTROL_WORK_MODE:
            ('duration', 'finish_date', 'evaluation_date'),
        LessonScenario.DIAGNOSTICS_MODE:
            ('duration', 'finish_date', 'diagnostics_text'),
    }

    default_messages = {
        'required': _(u'Это обязательное поле для режима занятия "{mode}"'),
        'dates': _(u'Дата окончания не может быть больше даты публикации'),
        'allow_time_limited_problems_no_blocks':
            _(u'В непрерывной олимпиаде нельзя объединять задачи в блоки'),
        'allow_time_limited_problems_training_only':
            _(u'Непрерывная олимпиада может быть только в тренировке'),
    }

    def __init__(self, fields_data, exception_class):
        """
        :type fields_data: dict | AbstractLessonScenario
        """
        self.fields_data = fields_data
        self.exception_class = exception_class

        if isinstance(fields_data, dict):
            self.get = lambda x: fields_data.get(x, None)
        else:
            self.get = lambda x: getattr(fields_data, x, None)

        if issubclass(exception_class, ValidationError):
            self.non_field_errors = NON_FIELD_ERRORS
        else:
            self.non_field_errors = api_settings.NON_FIELD_ERRORS_KEY

    @classmethod
    def validate(cls, fields_data, exception_class=ValidationError):
        """
        Валидирует данные модели AbstractLessonScenario и производных.

        В качестве `fields_data` можно передавать как словарь, так и
        экземпляр модели.

        :type fields_data: dict | AbstractLessonScenario
        """
        errors = {}
        self = cls(fields_data, exception_class)

        errors.update(self.validate_required_fields())
        errors.update(self.validate_dates())
        errors.update(
            self.validate_allow_time_limited_problems_visual_mode_no_blocks())
        errors.update(
            self.validate_allow_time_limited_problems_training_only())

        if errors:
            raise exception_class(errors)

    def validate_required_fields(self):
        """
        Проверяет, что у контрольной работы указаны время окончания,
        время публикации результата и длительность
        """
        errors = {}

        mode = self.get('mode')
        required_fields = self.required_fields.get(mode)
        if not required_fields:
            return errors

        for field in required_fields:
            if not self.get(field):  # empty field value
                errors[field] = self.default_messages['required'].format(
                    mode=self.lesson_modes[mode]
                )

        return errors

    def validate_dates(self):
        """
        Дата окончания не может быть больше даты публикации.
        """
        errors = {}

        finish_date = self.get('finish_date')
        evaluation_date = self.get('evaluation_date')

        if finish_date and evaluation_date:
            if finish_date > evaluation_date:
                errors[self.non_field_errors] = self.default_messages['dates']

        return errors

    def validate_allow_time_limited_problems_visual_mode_no_blocks(self):
        """
        При allow_time_limited_problems == True
        visual_mode не может принимать значение BLOCKS
        (Непрерывная олимпиада может быть только в тренировке)
        """
        errors = {}

        if (
                self.get('allow_time_limited_problems') and
                self.get(
                    'visual_mode') == LessonScenario.VisualModes.BLOCKS
        ):
            errors['visual_mode'] = (
                self.default_messages['allow_time_limited_problems_no_blocks']
            )

        return errors

    def validate_allow_time_limited_problems_training_only(self):
        """
        allow_time_limited_problems == True visual_mode
        может быть только при TRAINING_MODE
        (в непрерывной олимпиаде не даем объединять задачи в блоки)
        """
        errors = {}

        if (
                self.get('allow_time_limited_problems') and
                self.get('mode') != LessonScenario.TRAINING_MODE
        ):
            errors['allow_time_limited_problems'] = (
                self.default_messages[
                    'allow_time_limited_problems_training_only'
                ]
            )

        return errors
