from rest_condition import And, Or

from rest_framework import status, viewsets
from rest_framework.decorators import detail_route
from rest_framework.permissions import SAFE_METHODS, IsAuthenticated
from rest_framework.response import Response

from kelvin.accounts.permissions import IsStaff, IsTeacher, ObjectForOwner, ReadOnly
from kelvin.results.serializers import LessonResultSerializer

from .models import Lesson
from .serializers import LessonSerializer

# from kelvin.common.revisor import client as reviewer_client



class LessonViewSet(viewsets.ModelViewSet):
    """
        Просмотр, создание, изменение, удаление уроков
        """
    serializer_class = LessonSerializer

    LIST_OF_PREFETCH = (
        'lessonproblemlink_set',
        'lessonproblemlink_set__problem',
        'lessonproblemlink_set__problem__subject',
        'methodology',
    )
    LIST_OF_PREFETCH_EXPAND_PROBLEMS = (
        'lessonproblemlink_set__problem__resources',
        'lessonproblemlink_set__problem__meta',
    )
    LessonResultSerializer = LessonResultSerializer

    # здесь `And` позволяет смешать разные типы пермишенов
    permission_classes = [
        Or(ReadOnly, And(IsTeacher, ObjectForOwner), IsStaff)]

    @detail_route(methods=['post'], permission_classes=[IsAuthenticated])
    def answer(self, request, **kwargs):
        """
        Принимаем ответы на занятие
        """
        serializer = self.LessonResultSerializer(
            data={
                'lesson': kwargs[self.lookup_field],
                'answers': request.data,
            },
            context=self.get_serializer_context(),
        )
        serializer.is_valid(raise_exception=True)
        result = serializer.save()
        response = self.get_serializer().to_representation(self.get_object())
        response['attempt'] = serializer.to_representation(result)
        return Response(response, status=status.HTTP_201_CREATED)

    def get_queryset(self):
        """
        Делаем prefetch только для SAFE_METHODS иначе для put запросов будут
        возвращаться старые данные
        """
        queryset = Lesson.objects.all()
        if self.request.method in SAFE_METHODS:
            queryset = queryset.prefetch_related(*self.LIST_OF_PREFETCH)
            if self.request.query_params.get('expand_problems'):
                queryset = queryset.prefetch_related(
                    *self.LIST_OF_PREFETCH_EXPAND_PROBLEMS
                )
        return queryset

    def get_serializer_context(self):
        """
        Добавляем параметры из запроса в контекст сериализатора
        """
        ctx = super(LessonViewSet, self).get_serializer_context()
        ctx['expand_problems'] = self.request.query_params.get('expand_problems')
        ctx['hide_answers'] = self.request.query_params.get('hide_answers')
        return ctx

    def perform_create(self, serializer):
        """
        Задаем создателя занятия
        """
        serializer.save(owner=self.request.user)
