from builtins import object
from datetime import datetime
from operator import itemgetter

from django import forms
from django.conf import settings
from django.contrib.auth import get_user_model
from django.contrib.auth.models import Group
from django.core.exceptions import ValidationError
from django.utils.html import mark_safe
from django.utils.translation import ugettext_lazy as _

from kelvin.courses.models import Course
from kelvin.mail.models import EmailHistory, EmailTemplate, ScheduledEmail
from kelvin.mail.tasks import send_mails_task

User = get_user_model()


class EmailSendForm(forms.ModelForm):
    """
    Форма отправки почтовых рассылок
    """
    template = forms.ModelChoiceField(
        queryset=EmailTemplate.objects.all(),
        label=_(u'Шаблон'),
        required=False,
        help_text=_(u'Если не выбрать шаблон, '
                    u'отправится только поле «Текст письма»'),
    )
    subject = forms.CharField(
        label=_(u'Тема письма'),
        required=False,
    )
    html_text = forms.CharField(
        label=_(u'Текст письма'),
        widget=forms.Textarea,
        required=False,
    )
    email_list = forms.CharField(
        label=_(u'Список адресов'),
        required=False,
        widget=forms.Textarea,
    )
    course = forms.ModelChoiceField(
        queryset=Course.objects.all(),
        label=_(u'Курс, ученикам которого отправить письмо'),
        required=False,
    )
    force_send = forms.BooleanField(
        label=_(u'Отправить письмо в том числе и отписанным'),
        required=False,
        initial=False,
    )

    class Meta(object):
        model = EmailHistory
        fields = '__all__'

    def save(self, commit=True):
        """
        Отправляет рассылку с заданными в форме параметрами
        """
        # Сохраняем исходные данные для истории
        # Убираем id, если был, чтобы сделать новую запись
        self.cleaned_data.pop('id', None)
        history = super(EmailSendForm, self).save()
        template = self.cleaned_data.get('template')
        course = self.cleaned_data.get('course')

        # отправляем письма
        send_mails_task.delay(
            template and template.id,
            mark_safe(self.cleaned_data['html_text']),
            course and course.id,
            self.cleaned_data['subject'],
            self.cleaned_data.get('email_list'),
            self.cleaned_data.get('force_send'),
            history_id=history.id,
        )

        return self.cleaned_data['subject']


class ScheduledEmailForm(forms.ModelForm):
    """
    Форма для админки рассылки с расписанием
    """
    schedule = forms.CharField(
        widget=forms.Textarea,
        help_text=u'Для добавления новых дат отправки рассылки напишите '
                  u'дату-время в столбик в виде дд.мм.гггг-чч:мм '
                  u'(21.05.2016-19:00). '
                  u'"Отправлено" и "Ожидает" обозначают статус отправки в эту '
                  u'дату. '
                  u'Для новых дат рассылки их указывать не нужно')
    # для более естественного ввода и чтения
    SCHEDULE_DATE_FORMAT = '%d.%m.%Y-%H:%M'
    WAS_SENT = u'Отправлено'
    WAS_NOT_SENT = u'Ожидает'

    class Meta(object):
        model = ScheduledEmail
        fields = '__all__'

    def __init__(self, *args, **kwargs):
        """
        Изменяет строки даты-времени для большей читаемости
        """
        super(ScheduledEmailForm, self).__init__(*args, **kwargs)
        if kwargs.get('instance'):
            self.initial['schedule'] = u'\n'.join(
                u'{0} {1}'.format(
                    datetime.strptime(entry['date'], settings.DATETIME_FORMAT)
                            .strftime(self.SCHEDULE_DATE_FORMAT),
                    self.WAS_SENT if entry['sent'] else self.WAS_NOT_SENT,
                )
                for entry in kwargs['instance'].schedule
            )

    def clean_schedule(self):
        """
        Составляет список дат отправки. Сортирует по дате-времени, проставляет
        'sent': False, если не указано
        """
        schedule = []
        was_sent_lower = self.WAS_SENT.lower()
        for number, entry in enumerate(
                self.cleaned_data['schedule'].split('\n'), 1):
            space_split = entry.split()
            if len(space_split) > 2:
                raise ValidationError(
                    u'Укажите дату и отправленность (Отправлено или Ожидает) '
                    u'или только дату в строке {0}. '
                    u'Возможно, вы неправильно ввели дату?'
                    .format(number)
                )
            try:
                date = datetime.strptime(
                    space_split[0], self.SCHEDULE_DATE_FORMAT)
            except ValueError:
                raise ValidationError(
                    u'Неправильная дата в строке {0}. Укажите в формате '
                    u'дд.мм.гггг-чч:мм'.format(number)
                )
            sent = (
                space_split[1].lower() == was_sent_lower
                if len(space_split) > 1
                else False
            )

            schedule.append({'date': date, 'sent': sent})
        schedule = sorted(schedule, key=itemgetter('date'))
        for entry in schedule:
            entry['date'] = entry['date'].strftime(
                settings.DATETIME_FORMAT)
        return schedule
