from datetime import datetime

from django.conf import settings
from django.conf.urls import url
from django.contrib import admin
from django.core.urlresolvers import reverse
from django.http import HttpResponseRedirect
from django.shortcuts import get_object_or_404
from django.template.response import TemplateResponse
from django.utils.translation import ugettext_lazy as _

from kelvin.mail.admin.forms import EmailSendForm, ScheduledEmailForm
from kelvin.mail.models import EmailHistory, EmailTemplate, ScheduledEmail


class EmailTemplateAdmin(admin.ModelAdmin):
    """
    Админка сертификатов по мероприятию
    """
    change_list_template = 'mail/admin/templates_change_list.html'
    mail_sender_template = 'mail/admin/mail_sender_form.html'

    search_fields = (
        'name',
    )

    def get_urls(self):
        """
        Добавляет урл для отправки рассылок
        """
        return [
            url(r'mail_sender/$',
                self.admin_site.admin_view(self.send_mail),
                name='mail_emailtemplate_mail_sender'),
        ] + super(EmailTemplateAdmin, self).get_urls()

    def send_mail(self, request):
        """
        Форма отправки рассылок
        """
        context = {
            'title': _(u'Почтовая рассылка'),
            'opts': self.model._meta
        }
        if request.method == 'POST':
            form = EmailSendForm(data=request.POST)
            context['form'] = form

            if form.is_valid():
                subject = form.save()
                self.message_user(
                    request,
                    _(u'Начата отправка рассылки «{0}»'.format(subject))
                )
                if "_continue" in request.POST:
                    return TemplateResponse(
                        request,
                        self.mail_sender_template,
                        context,
                    )
                else:
                    return HttpResponseRedirect(
                        reverse('admin:{0}_{1}_changelist'.format(
                            EmailTemplate._meta.app_label,
                            EmailTemplate._meta.model_name
                        ), current_app=self.admin_site.name)
                    )
            else:
                return TemplateResponse(
                    request,
                    self.mail_sender_template,
                    context,
                )

        # Рассылка на основе ранее отправленной
        history_id = request.GET.get('history_id')
        if history_id:
            history = get_object_or_404(EmailHistory, id=history_id)
            context['form'] = EmailSendForm(instance=history)
            return TemplateResponse(
                request,
                self.mail_sender_template,
                context,
            )

        # Рассылка на основе списка адресов
        emails = request.session.pop('mail_sender__emails', None)
        initial = {'email_list': '\n'.join(emails)} if emails else {}

        context['form'] = EmailSendForm(initial=initial)
        return TemplateResponse(
            request,
            self.mail_sender_template,
            context,
        )


class EmailHistoryAdmin(admin.ModelAdmin):
    list_display = (
        'id',
        'subject',
        'template',
        'group',
        'course',
        'date_sent',
    )
    list_select_related = ('group', 'course',)
    change_form_template = 'mail/admin/history_change_form.html'

    def has_add_permission(self, request):
        """Не даем добавлять в историю"""
        return False


class ScheduledEmailAdmin(admin.ModelAdmin):
    list_display = (
        'id',
        'name',
        'subject',
        'course',
        'group',
        'get_last_sent_date',
        'get_next_send_date',
    )
    form = ScheduledEmailForm
    raw_id_fields = ('course', )

    def get_last_sent_date(self, obj):
        """
        Извлекает последнюю дату отправки. Возвращает None, если ничего
        не отправлялось
        """
        for schedule_entry in reversed(obj.schedule):
            if schedule_entry['sent']:
                return datetime.strptime(schedule_entry['date'],
                                         settings.DATETIME_FORMAT)
    get_last_sent_date.short_description = u'Дата последней отправки'

    def get_next_send_date(self, obj):
        """
        Извлекает дату следующей запланированной отправки. Возвращает None,
        если больше ничего отправлять не нужно
        """
        for schedule_entry in obj.schedule:
            if not schedule_entry['sent']:
                return datetime.strptime(schedule_entry['date'],
                                         settings.DATETIME_FORMAT)
    get_next_send_date.short_description = u'Дата следующей отправки'

admin.site.register(EmailTemplate, EmailTemplateAdmin)
admin.site.register(EmailHistory, EmailHistoryAdmin)
admin.site.register(ScheduledEmail, ScheduledEmailAdmin)
