from builtins import object

from premailer import transform

from django.conf import settings
from django.contrib.auth import get_user_model
from django.contrib.auth.models import Group
from django.core.mail import EmailMultiAlternatives
from django.db import models
from django.template import Context, Template
from django.utils.encoding import python_2_unicode_compatible
from django.utils.translation import ugettext_lazy as _

from kelvin.common.fields import JSONField
from kelvin.courses.models import Course


@python_2_unicode_compatible
class EmailTemplate(models.Model):
    """
    Шаблон письма email-рассылки
    """
    name = models.CharField(
        verbose_name=_('Название шаблона'),
        max_length=255,
    )
    template = models.TextField(verbose_name=_('Шаблон письма'))

    def __str__(self):
        return self.name

    def render(self, context_dict):
        """
        Рендеринг шаблона стандартными джанговыми средствами и обработка
        премейлером
        """
        template = Template(self.template)
        context = Context(context_dict)

        return transform(template.render(context))

    class Meta(object):
        verbose_name = _('Шаблон письма')
        verbose_name_plural = _('Шаблоны письма')


class EmailHistory(models.Model):
    """
    История отправленных рассылок
    """
    template = models.ForeignKey(
        EmailTemplate,
        verbose_name=_('Шаблон письма'),
        blank=True,
        null=True,
    )
    subject = models.CharField(
        verbose_name=_('Тема'),
        max_length=255,
        blank=True,
    )
    html_text = models.TextField(
        verbose_name=_('Текст письма'),
        blank=True,
    )
    email_list = models.TextField(
        verbose_name=_('Адресаты'),
        blank=True,
    )
    group = models.ForeignKey(
        Group,
        verbose_name=_('Группа дополнительных адресатов'),
        blank=True,
        null=True,
    )
    course = models.ForeignKey(
        Course,
        verbose_name=_('Курс, ученикам которого сделана рассылка'),
        blank=True,
        null=True,
    )
    date_sent = models.DateTimeField(
        verbose_name='Дата и время отправки',
        auto_now_add=True,
    )
    force_send = models.BooleanField(
        verbose_name='Была отправлена в том числе отписанным',
        default=False,
    )
    # Шаблон тега для отслеживания рассылки
    TAG_TEMPLATE = '{prefix}-{history_id}'

    class Meta(object):
        verbose_name = _('Отправленная рассылка')
        verbose_name_plural = _('Отправленные рассылки')


class ScheduledEmail(models.Model):
    """
    Почтовая рассылка с расписанием
    """
    name = models.CharField(
        verbose_name=_('Имя рассылки'),
        max_length=255,
        blank=True,
    )
    template = models.ForeignKey(
        EmailTemplate,
        verbose_name=_('Шаблон письма'),
        blank=True,
        null=True,
    )
    subject = models.CharField(
        verbose_name=_('Тема'),
        max_length=255,
        blank=True,
    )
    html_text = models.TextField(
        verbose_name=_('Текст письма'),
        blank=True,
    )
    email_list = models.TextField(
        verbose_name=_('Адресаты'),
        blank=True,
    )
    group = models.ForeignKey(
        Group,
        verbose_name=_('Группа дополнительных адресатов'),
        blank=True,
        null=True,
    )
    course = models.ForeignKey(
        Course,
        verbose_name=_('Курс, ученикам которого сделать рассылку'),
        blank=True,
        null=True,
    )
    schedule = JSONField(
        verbose_name=_('Расписание отправки'),
        blank=True,
        default=[],
    )

    class Meta(object):
        verbose_name = _('Рассылка с расписанием')
        verbose_name_plural = _('Рассылки с расписанием')


def send_mails(template, html_text, course, subject, email_list, force_send=False, history_id=None):
    """
    Определяет что и кому отправить

    :param template: шаблон письма
    :type template: EmailTemplate
    :param html_text: текст письма
    :param course: курс, ученикам которого или ученикам копий которого будет
                   отправлено письмо
    :type course: Course
    :param subject: тема письма
    :param email_list: адресаты письма, список адресов, записанных через
                       перенос строки
    :param force_send: отправить в том числе и отписанным
    :type force_send: bool
    """
    if history_id:
        tag = EmailHistory.TAG_TEMPLATE.format(
            prefix=settings.EMAIL_TAG_PREFIX, history_id=history_id)

    User = get_user_model()

    subject = subject or template.name
    recipients = set()

    if course:
        # Для оригинальных курсов рассылку делаем по ученикам копий
        if not course.copy_of:
            for copy in Course.objects.filter(
                    copy_of=course).prefetch_related('students'):
                recipients.update(copy.students.all())
        else:
            recipients.update(course.students.all())

    # Пользователей, указанных почтой, пытаемся найти
    if email_list:
        email_set = {
            email for email in email_list.replace('\r\n', '\n').split('\n')
            if email
        }
        users_by_mail = User.objects.filter(
            email__in=email_set, email__isnull=False)
        recipients.update(users_by_mail)

        # Тем, кого найти не удалось, отправляем неперсонализированное письмо
        generic_message = template.render(
            {'inner': html_text, 'course': course}) if template else html_text
        unidentified_recipients = list(email_set - {
            user.email for user in users_by_mail})

        mail = EmailMultiAlternatives(
            subject=subject,
            from_email=settings.DEFAULT_FROM_EMAIL,
            bcc=unidentified_recipients,
        )
        mail.attach_alternative(generic_message, 'text/html')
        if history_id:
            mail.extra_headers['X-Mailgun-Tag'] = tag
        mail.send()

    def get_mail_from_recepient(recepient):
        """
        Пробуем по юзеру получить емейл.
        Сперва из пользователя, потом из профиля несовершеннолетнего.
        """

        if recepient.email:
            return recepient.email

        return None

    # Пользователям рендерим отдельные сообщения с учетом их данных
    for recipient in recipients:
        # FIXME: оторвать этот код
        email = get_mail_from_recepient(recipient)
        if not email:
            continue

        # Не отправляем письма отписанным пользователям
        # Но отправляем, если есть `force_send`
        if not force_send and recipient.unsubscribed:
            continue
        message = template.render({
            'inner': html_text,
            'course': course,
            'user': recipient,
        }) if template else html_text

        mail = EmailMultiAlternatives(
            subject=subject,
            from_email=settings.DEFAULT_FROM_EMAIL,
            bcc=[email],
        )
        if history_id:
            mail.extra_headers['X-Mailgun-Tag'] = tag
        mail.attach_alternative(message, 'text/html')
        mail.send()

    return subject
