import logging
from datetime import datetime

from django.conf import settings
from django.contrib.auth.models import Group
from django.utils.html import mark_safe

from kelvin.celery import app
from kelvin.courses.models import Course
from kelvin.mail.models import EmailHistory, EmailTemplate, ScheduledEmail, send_mails

logger = logging.getLogger(__name__)


@app.task()
def check_scheduled_email():
    """
    Проверяет все рассылки по расписанию и отправляет те, которые должны были
    быть отправлены
    """
    now = datetime.now()
    for scheduled_email in ScheduledEmail.objects.all():
        for i, item in enumerate(scheduled_email.schedule):
            if item['sent']:
                continue
            date = datetime.strptime(item['date'], settings.DATETIME_FORMAT)
            if date <= now:
                history = EmailHistory.objects.create(
                    template=scheduled_email.template,
                    html_text=scheduled_email.html_text,
                    course=scheduled_email.course,
                    subject=scheduled_email.subject,
                    email_list=scheduled_email.email_list,
                )
                send_mails(
                    scheduled_email.template,
                    mark_safe(scheduled_email.html_text),
                    scheduled_email.course,
                    scheduled_email.subject,
                    scheduled_email.email_list,
                    history_id=history.id,
                )
                scheduled_email.schedule[i]['sent'] = True
                scheduled_email.save()


@app.task()
def send_mails_task(template_id, html_text, course_id, subject, email_list,
                    force_send=False, history_id=None):
    """
    Отправляет рассылку, выполняется на Celery для того, чтоб работа с базой
    (получение пользователей по email_list) выполнялась асинхронно.

    Информация о параметрах: см. документацию kelvin.mail.models.send_mails

    Параметры *_id — id соответствующих объе
    """
    template = template_id and EmailTemplate.objects.get(id=template_id)
    course = course_id and Course.objects.get(id=course_id)

    send_mails(template, html_text, course, subject,
               email_list, group, force_send, history_id)
