from builtins import object

from django.conf import settings
from django.db import models
from django.utils.encoding import python_2_unicode_compatible
from django.utils.translation import ugettext_lazy as _

from kelvin.common.fields import JSONField
from kelvin.common.model_mixins import TimeStampMixin
from kelvin.courses.models import Course
from kelvin.news.schemas import InfoFieldSchema, info_schema_validator


@python_2_unicode_compatible
class News(TimeStampMixin, models.Model):
    """
    Новость
    """

    class Status(object):
        DRAFT = 0
        PUBLISHED = 1

        MODES = (
            (DRAFT, 'черновик'),
            (PUBLISHED, 'опубликован'),
        )

        DEFAULT = DRAFT

    title = models.CharField(
        verbose_name=_('Заголовок новости'),
        max_length=512,
    )
    summary = models.TextField(
        verbose_name=_('Краткое описание новости'),
        blank=True,
    )
    content = models.TextField(
        verbose_name=_('Содержание новости (markdown)'),
    )
    status = models.SmallIntegerField(
        verbose_name=_('Статус новости'),
        choices=Status.MODES,
        default=Status.DEFAULT,
    )
    owner = models.ForeignKey(
        settings.AUTH_USER_MODEL,
        verbose_name=_('Автор новости'),
    )
    publication_date = models.DateTimeField(
        verbose_name=_('Дата публикации'),
        null=True,
    )

    courses = models.ManyToManyField(
        Course,
        verbose_name=_('Курсы, к которым привязаны новости'),
        through='NewsCourseLink',
        related_name='publications',
        blank=True,
    )

    info = JSONField(
        verbose_name=_('Дополнительная информация для новости'),
        help_text=_(
            'Объект с дополнительной информацией для новости, например:'
            '{}'.format(InfoFieldSchema.ADMIN_HINT)
        ),
        validators=[info_schema_validator],
        default=dict,
    )

    class Meta(object):
        verbose_name = _('Новость')
        verbose_name_plural = _('Новости')

    def __str__(self):
        return 'News {0}'.format(self.id)


@python_2_unicode_compatible
class NewsCourseLink(models.Model):
    """
    Связь новости с курсами
    """

    news = models.ForeignKey(
        News,
        verbose_name=_('Новость'),
    )
    course = models.ForeignKey(
        Course,
        verbose_name=_('Курс'),
    )

    class Meta(object):
        verbose_name = _('Связь новости с курсом')
        verbose_name_plural = _('Связи новостей с курсами')

    def __str__(self):
        return 'News {0}, Course {1}, ID: {2}'.format(
            self.news_id, self.course_id, self.id)
