from builtins import object

from django.utils import timezone

from rest_framework import serializers
from rest_framework.compat import authenticate

from kelvin.common.serializer_fields import JSONField
from kelvin.common.serializer_mixins import SerializerManyToManyMixin
from kelvin.common.serializers import ManyToManyListSerializer
from kelvin.news.models import News, NewsCourseLink


class CourseNewsSerializer(serializers.ModelSerializer):
    """
    Сериализатор привязки новости к курсу
    """

    class Meta(object):
        model = NewsCourseLink
        fields = ('id', 'course')
        list_serializer_class = ManyToManyListSerializer


class NewsSerializer(SerializerManyToManyMixin,
                     serializers.ModelSerializer):
    """
    Сериализатор новости
    """

    m2m_update_fields = {'courses': 'news'}
    courses = CourseNewsSerializer(
        source='newscourselink_set',
        required=False,
        many=True,
    )
    info = JSONField(required=False)

    class Meta(object):
        model = News
        fields = (
            'id',
            'title',
            'summary',
            'content',
            'status',
            'owner',
            'publication_date',
            'courses',
            'info',
        )
        read_only_fields = (
            'owner',
            'publication_date',
        )

    default_error_messages = {
        'no user_id in context': u'В контексте нет id пользователя',
    }

    def validate(self, attrs):
        if 'owner' not in attrs:
            if (
                'request' in self.context and
                authenticate(self.context['request'].user)
            ):
                attrs['owner'] = self.context['request'].user
            else:
                self.fail('no user_id in context')
        return attrs

    def create(self, validated_data):
        if validated_data.get('status') == News.Status.PUBLISHED:
            validated_data['publication_date'] = timezone.now()
        return super(NewsSerializer, self).create(validated_data)

    def update(self, instance, validated_data):
        if (
            instance.status == News.Status.DRAFT and
            validated_data.get('status') == News.Status.PUBLISHED
        ):
            validated_data['publication_date'] = timezone.now()
        return super(NewsSerializer, self).update(instance, validated_data)
