from builtins import object

from mptt.models import MPTTModel, TreeForeignKey

from django.contrib.postgres.fields.array import ArrayField
from django.db import models
from django.utils.encoding import python_2_unicode_compatible
from django.utils.translation import ugettext_lazy as _

from kelvin.common.model_mixins import TimeStampMixin
from kelvin.group_levels.models import GroupLevel
from kelvin.subjects.models import Subject, Theme


@python_2_unicode_compatible
class ProblemMetaExam(models.Model):
    """
    Дополнительные поля для экзаменов, привязанных к информации о задаче
    """
    exam = models.ForeignKey('Exam', verbose_name=_('Экзамен'))
    problem_meta = models.ForeignKey('ProblemMeta',
                                     verbose_name=_('Мета-информация задачи'))
    number = models.CharField(_('Номер в экзамене'), max_length=255)

    class Meta(object):
        verbose_name = _('Связь мета-информации и экзамена')
        verbose_name_plural = _('Связь мета-информации и экзаменов')
        ordering = ('exam__id',)
        unique_together = ('exam', 'problem_meta')

    def __str__(self):
        return '{0} №{1}'.format(self.exam, self.number)


@python_2_unicode_compatible
class ProblemMeta(TimeStampMixin, models.Model):
    """
    Информация о задаче
    """
    DIFFICULTY_LIGHT = 1
    DIFFICULTY_MEDIUM = 2
    DIFFICULTY_HARD = 3
    DIFFICULTIES = (
        (DIFFICULTY_LIGHT, 'Простой'),
        (DIFFICULTY_MEDIUM, 'Средний'),
        (DIFFICULTY_HARD, 'Сложный'),
    )

    difficulty = models.IntegerField(
        verbose_name=_('Сложность задачи'),
        choices=DIFFICULTIES,
    )
    tags = ArrayField(
        models.CharField(max_length=255),
        verbose_name=_('Теги задачи'),
        blank=True,
        null=True,
    )
    main_theme = models.ForeignKey(
        Theme, verbose_name=_('Основная тема'), related_name='main_theme',
        blank=True, null=True,
    )
    additional_themes = models.ManyToManyField(
        Theme,
        verbose_name=_('Дополнительные темы'),
        blank=True,
    )
    group_levels = models.ManyToManyField(
        GroupLevel,
        verbose_name=_('Классы'),
        blank=True,
    )
    skills = models.ManyToManyField('Skill', verbose_name=_('Навыки'),
                                    blank=True)
    exams = models.ManyToManyField('Exam', through=ProblemMetaExam,
                                   verbose_name=_('Экзамены'), blank=True)

    def __str__(self):
        """
        Представление метаинформации в виде строки
        """
        return 'Id: {0}, Difficulty: {1}, Tags: {2}'.format(
            self.id, self.get_difficulty_display(),
            ', '.join(self.tags) if self.tags else '<None>',
        )


@python_2_unicode_compatible
class Exam(models.Model):
    """
    Модель экзамена
    """
    LEVEL_BASIC = 1
    LEVEL_ADVANCED = 2
    LEVEL_MISSING = 3
    LEVELS = (
        (LEVEL_BASIC, _('Базовый')),
        (LEVEL_ADVANCED, _('Профильный')),
        (LEVEL_MISSING, _('Отсутствует')),
    )
    name = models.CharField(
        verbose_name=_('Название экзамена'),
        max_length=255,
    )
    year = models.IntegerField(
        verbose_name=_('Год'),
    )
    level = models.IntegerField(
        verbose_name=_('Уровень экзамена'),
        choices=LEVELS,
        default=LEVEL_MISSING,
    )

    class Meta(object):
        verbose_name = _('Экзамен')
        verbose_name_plural = _('Экзамены')

    def __str__(self):
        return '{0} ({1})'.format(self.name, self.year)


@python_2_unicode_compatible
class Skill(MPTTModel):
    """
    Справочник навыков
    """
    name = models.CharField(_('Название'), max_length=255)
    subject = models.ForeignKey(Subject)
    parent = TreeForeignKey('self', null=True, blank=True,
                            related_name='children', db_index=True)

    class Meta(object):
        verbose_name = _('Навык')
        verbose_name_plural = _('Навыки')

    def __str__(self):
        """Выводит предмет и название навыка"""
        return '{0}: {1}'.format(self.subject, self.name)
