from builtins import object

from rest_framework import serializers

from kelvin.common.serializer_mixins import (
    DateUpdatedFieldMixin, SerializerForeignKeyMixin, SerializerManyToManyMixin, SetFieldsMixin, SkipNullsMixin,
)
from kelvin.common.serializers import ManyToManyListSerializer
from kelvin.group_levels.serializers import GroupLevelListField
from kelvin.problem_meta.models import Exam, ProblemMeta, ProblemMetaExam, Skill
from kelvin.subjects.models import Theme
from kelvin.subjects.serializers import SubjectSerializer, ThemeSerializer


class ProblemMetaExamSerializer(serializers.ModelSerializer):
    """
    Сериализатор поля `exams` для метаинформации
    """
    id = serializers.IntegerField(required=False)

    class Meta(object):
        model = ProblemMetaExam
        fields = ('id', 'exam', 'number')
        list_serializer_class = ManyToManyListSerializer


class ProblemMetaSerializer(SetFieldsMixin,
                            SkipNullsMixin,
                            DateUpdatedFieldMixin,
                            SerializerManyToManyMixin,
                            serializers.ModelSerializer):
    """
    Сериализатор модели метаинформации задачи
    """
    additional_themes = serializers.PrimaryKeyRelatedField(
        queryset=Theme.objects.all(),
        required=False,
        many=True,
    )

    group_levels = GroupLevelListField(required=False)
    exams = ProblemMetaExamSerializer(source='problemmetaexam_set', many=True,
                                      required=False)

    m2m_update_fields = {'exams': 'problem_meta'}

    class Meta(object):
        model = ProblemMeta
        fields = (
            'id',
            'date_updated',
            'difficulty',
            'tags',
            'main_theme',
            'additional_themes',
            'exams',
            'group_levels',
            'skills',
        )
        read_only_fields = (
            'date_updated',
        )
        extra_kwargs = {'skills': {'required': False}}


class ProblemMetaMainThemeSerializer(ProblemMetaSerializer,
                                     SerializerForeignKeyMixin):
    """
    Сериализатор метаинформации с сериализацией главной темы
    """

    fk_update_fields = ['main_theme']
    main_theme = ThemeSerializer()


class ProblemMetaIdDateSerializer(DateUpdatedFieldMixin,
                                  serializers.ModelSerializer):
    """
    Сериализатор модели метаинформации задачи для случаев, когда нужно
    представить ее в коротком виде (id и время обновления)
    """

    class Meta(object):
        model = ProblemMeta
        fields = (
            'id',
            'date_updated',
        )


class ProblemMetaExpansionAwareSerializer(ProblemMetaSerializer):
    """
    Сериализатор модели метаинформации, сериализация которого зависит
    от параметра `expand_meta` в запросе
    """
    def to_representation(self, meta):
        """
        Сериализация метаинформации задачи. Включает все поля, если
        в запросе есть параметр `expand_meta`, и только id и время обновления,
        если в запросе нет параметра либо запрос отсутствует
        """
        request = self.context.get('request')
        expand = request.query_params.get('expand_meta') if request else False
        expand_main_theme = (self.context.get('expand_main_theme') if
                             self.context else False)
        if expand_main_theme:
            return ProblemMetaMainThemeSerializer().to_representation(meta)

        if expand:
            return super(ProblemMetaExpansionAwareSerializer,
                         self).to_representation(meta)
        else:
            return ProblemMetaIdDateSerializer().to_representation(meta)


class ExamSerializer(serializers.ModelSerializer):
    """
    Сериалайзер модели Экзамен
    """
    class Meta(object):
        model = Exam


class SkillSerializer(serializers.ModelSerializer):
    """
    Сериализатор для модели Навыки
    """
    subject = SubjectSerializer(read_only=True)

    class Meta(object):
        model = Skill
        fields = ('id', 'subject', 'name')
        read_only_fields = ('id',)
