from builtins import map, object, str

from past.utils import old_div

from kelvin.resources.models import Resource

DRAGIMAGE_FIELD_KEYS = {
    'id',
    'x',
    'y',
    'disabled',
    'choice',
    'gravity',
}


class MarkupConverterError(Exception):
    """
    Обобщающее исключение для ошибок конвертации
    """
    pass


class MarkupConverter(object):
    """
    Класс для конвертирования форматов разметки Учебника в новые универсальные
    """
    CONVERT_METHOD_FORMAT = 'convert_{0}'

    def __init__(self, **args):
        self.params = args

    @staticmethod
    def _search_container_by_marker_id(marker_id, containers):
        """
        Возвращает индекс первого контейнера, который содержит подстроку
        '{marker:<marker_id>}' или None, если не найден
        """
        found_indices = [
            i for i, container in enumerate(containers) if
            '{{marker:{0}}}'.format(marker_id) in container.get('content', '')
        ]
        return found_indices[0] if found_indices else None

    def convert(self, markup):
        """
        Конвертирует разметку задачи для каждого конвертируемого маркера:
        выделяет поле `content` для layout-а, проверку и ответ,
        каким образом - зависит от каждой конвертации.
        Конвертациям на вход идет содержимое поля `content`, конвертации
        могут изменять его.

        :raises: `MarkupConverterError` из методов-конвертеров, а также
        если не было конвертировано ни одного маркера
        """
        converted = 0
        for item in markup['layout']:
            if item['kind'] != 'marker':
                continue
            marker_converter = self.CONVERT_METHOD_FORMAT.format(
                item['content'].get('type'))
            if not hasattr(self, marker_converter):
                continue
            new_content, check, answer = getattr(self, marker_converter)(
                item['content'],
                markup['answers'].get(str(item['content']['id'])),
            )
            item['content'] = new_content
            if check is not None:
                markup['checks'][str(new_content['id'])] = check
            if answer is not None:
                markup['answers'][str(new_content['id'])] = answer
            converted += 1

        if not converted:
            raise MarkupConverterError(
                u'Не было конвертировано ни одного маркера')

        return markup

    def convert_coreplaceimages(self, marker_data, answer_data):
        """
        Преобразование перетаскивалок
        """
        screen_width = 1280.
        screen_height = 800.
        indent = 10

        # берем ресурсы картинкок
        main_image = Resource.objects.get(
            id=marker_data['options']['main_image'])
        choice_resources = {
            resource.id: resource
            for resource in Resource.objects.filter(
                id__in=[choice['image'] for choice
                        in marker_data['options']['answers']],
            )
        }
        field = Resource.objects.get(id=marker_data['options']['field'])

        # считаем координаты и ширину объектов
        scale_factor = old_div(screen_width, main_image.image_width)
        if self.params.get('width_1'):
            main_image_width = 1.
        else:
            max_choice_height = max(
                choice.image_height for choice in choice_resources.values()
            )
            if screen_height < (
                (max_choice_height + main_image.image_height) * scale_factor +
                    3 * indent):
                scale_factor = old_div((screen_height - 3 * indent), (
                    max_choice_height + main_image.image_height))
            if screen_width < scale_factor * sum(
                [choice.image_width + indent
                 for choice in choice_resources.values()]):
                scale_factor = old_div(screen_width, sum(
                    [choice.image_width + indent
                     for choice in choice_resources.values()]
                ))

            main_image_width = (main_image.image_width *
                                scale_factor / screen_width)

        # словарь с новыми и старыми идентификаторами филдов (старые могут
        # быть строки типа `1_0`, новые только целые числа)
        field_ids = {}
        for field_ in marker_data['options']['fields']:
            try:
                field_ids[field_['id']] = int(field_['id'])
            except ValueError:
                # сначала добавляем в словарь все правильные целые
                pass
        for field_ in marker_data['options']['fields']:
            # добавляем в словарь старые нецелые идентификаторы
            if field_['id'] not in field_ids:
                field_ids[field_['id']] = (
                    max(field_ids.values()) + 1 if field_ids else 1)

        # пересчитываем координаты полей
        fields = []
        places_for_choices = {}
        for field_ in marker_data['options']['fields']:
            field_['x'] *= main_image_width
            field_['y'] *= (old_div(float(main_image.image_height),
                                    main_image.image_width)) * main_image_width
            if field_.get('container') and 'answer' in field_:
                # поле, на которое надо поставить чойс
                places_for_choices[field_['answer']] = field_
            else:
                field_['id'] = field_ids[field_['id']]
                fields.append(field_)

        # оставляем в полях только разрешённые ключи
        fields = [{
            key: field_[key]
            for key in DRAGIMAGE_FIELD_KEYS.intersection(list(field_.keys()))
        } for field_ in fields]

        # добавляем отступ от главной картинки
        choices_y = ((main_image.image_height + indent) * scale_factor /
                     screen_width)
        unpinned_choices_width = 0  # ширина чойсов, располагаемых внизу
        unpinned_choices_num = 0  # число чойсов, расположенных внизу
        for choice in marker_data['options']['answers']:
            if choice['image'] not in places_for_choices:
                unpinned_choices_width += (
                    choice_resources[choice['image']].image_width *
                    scale_factor / screen_width
                )
                unpinned_choices_num += 1
        unpinned_chices_indent = old_div((1. - unpinned_choices_width), float(
            unpinned_choices_num + 1))
        choice_x = unpinned_chices_indent
        choices = []
        for i, choice in enumerate(marker_data['options']['answers'], 1):
            new_choice = {
                'id': i,
                'value': choice['value'],
                'resource_id': choice['image'],
                'width': (choice_resources[choice['image']].image_width *
                          scale_factor / screen_width),
            }
            if choice['image'] in places_for_choices:
                place = places_for_choices[choice['image']]
                # ставим чойсы, чтобы центры были в координатах старой разметки

                width = choice_resources[choice['image']].image_width
                new_choice['x'] = (
                    place['x'] - width * scale_factor / screen_width / 2.
                )

                height = choice_resources[choice['image']].image_height
                new_choice['y'] = (
                    place['y'] - height * scale_factor / screen_width / 2.
                )
            else:
                new_choice['x'] = choice_x
                new_choice['y'] = choices_y
                choice_x += new_choice['width'] + unpinned_chices_indent
            choices.append(new_choice)
        choice_by_resource_id = {choice['resource_id']: choice
                                 for choice in choices}

        # формируем новую разметку
        new_marker_data = {
            'id': int(marker_data['id']),
            'type': 'dragimage',
            'options': {
                'main_image': {
                    'resource_id': marker_data['options']['main_image'],
                    'x': 0,
                    'y': 0,
                    'width': main_image_width,
                },
                'fields': fields,
                'choices': choices,
                'multiple_choices': marker_data['options']['multiple_answers'],
                'gravity_radius': (
                    max([field.image_width, field.image_height]) / 2. * scale_factor / screen_width
                ),
            },
        }

        # центрируем `main_image` с филдами и прикрепленными чойсами
        if new_marker_data['options']['main_image']['width'] < 1:
            shift = (
                old_div((1 - new_marker_data['options']['main_image']['width']), 2.)
            )
            new_marker_data['options']['main_image']['x'] += shift
            for field_ in fields:
                field_['x'] += shift
            for choice in choices:
                if choice['resource_id'] in places_for_choices:
                    choice['x'] += shift

        # преобразуем проверку

        def convert_test(test):
            """Преобразование старой проверки в новую"""
            type_ = test['type']
            if type_ == 'SUM':
                if len(test['ids']) > 1:
                    return {
                        'type': 'EQUAL',
                        'sources': [
                            {
                                'type': 'SUM',
                                'sources': [
                                    {
                                        'type': 'FIELD',
                                        'source': field_ids[id_],
                                    }
                                    for id_ in test['ids']
                                ],
                            },
                            {
                                'type': 'NUMBER',
                                'source': test['test_value'],
                            },
                        ],
                    }
                else:
                    test_field_id = field_ids[test['ids'][0]]
                    if len(fields) == 1 and test_field_id != fields[0]['id']:
                        # исправление неправильного идентификатора в ответе
                        # это ошибка старой разметки
                        test_field_id = fields[0]['id']
                    return {
                        'type': 'EQUAL',
                        'sources': [
                            {
                                'type': 'FIELD',
                                'source': test_field_id,
                            },
                            {
                                'type': 'NUMBER',
                                'source': test['test_value'],
                            },
                        ],
                    }
            elif type_ == 'DIFFERENT':
                return {
                    'type': type_,
                    'sources': [
                        {'type': 'FIELD', 'source': field_ids[id_]}
                        for id_ in test['ids']
                    ],
                }
            elif type_ in ['MORE', 'EQMORE', 'LESS', 'EQLESS']:
                return {
                    'type': type_,
                    'sources': [
                        {
                            'type': 'FIELD',
                            'source': field_ids[test['ids'][0]],
                        },
                        {'type': 'NUMBER', 'source': test['test_value']},
                    ],
                }
            elif type_ == 'SUM_EQ':
                return {
                    'type': 'EQUAL',
                    'sources': [
                        {
                            'type': 'SUM',
                            'sources': [
                                {'type': 'FIELD', 'source': field_ids[id_]}
                                for id_ in test['ids']
                            ],
                        },
                        {
                            'type': 'SUM',
                            'sources': [
                                {'type': 'FIELD', 'source': field_ids[id_]}
                                for id_ in test['compare_to_ids']
                            ],
                        },
                    ],
                }
            elif type_ in ['SUM_MORE', 'SUM_EQMORE', 'SUM_LESS', 'SUM_EQLESS']:
                return {
                    'type': type_[4:],
                    'sources': [
                        {
                            'type': 'SUM',
                            'sources': [
                                {'type': 'FIELD', 'source': field_ids[id_]}
                                for id_ in test['ids']
                            ],
                        },
                        {'type': 'FIELD', 'source': test['test_value']},
                    ],
                }
            elif type_ == 'DIFFERENT_SETS':
                return {
                    'type': 'UNIQUE_ITEMS',
                    'sources': [
                        {
                            'type': 'UNION',
                            'sources': [
                                {'type': 'FIELD', 'source': field_ids[id_]}
                                for id_ in test['ids']
                            ],
                        },
                        {
                            'type': 'UNION',
                            'sources': [
                                {'type': 'FIELD', 'source': field_ids[id_]}
                                for id_ in test['compare_to_ids']
                            ],
                        },
                    ],
                }
            elif type_ == 'PERMITTED_ANSWERS':
                return {
                    'type': 'IS_SUBSET',
                    'sources': [
                        {
                            'type': 'UNION',
                            'sources': [
                                {'type': 'FIELD', 'source': field_ids[id_]}
                                for id_ in test['ids']
                            ],
                        },
                        {
                            'type': 'UNION',
                            'sources': [
                                {
                                    'type': 'NUMBER',
                                    'source': (
                                        choice_by_resource_id[answer]['value']
                                    ),
                                }
                                for answer in test['answers']
                            ]
                        }
                    ],
                }
            else:
                raise MarkupConverterError(
                    u'Не поддерживается тип проверки {1} в маркере {0}'
                    .format(marker_data['id'], type_)
                )

        if len(marker_data['options']['tests']) > 1:
            check = {
                'type': 'AND',
                'sources': list(map(convert_test, marker_data['options']['tests']))
            }
        else:
            check = convert_test(marker_data['options']['tests'][0])

        return new_marker_data, check, {}
