from django.contrib import admin
from django.db.models import Q
from django.utils.translation import ugettext_lazy as _


class ImportedProblemsFilter(admin.SimpleListFilter):
    """
    Фильтрация по импортированным задачам
    """
    title = u'Импортированные задачи'
    parameter_name = 'imported_from'
    LOOKUPS = (
        ('1', u'Из Учебы'),
    )

    def queryset(self, request, queryset):
        """
        Фильтрация значений по выбранному фильтру
        """
        if self.value() == self.LOOKUPS[0][0]:
            return queryset.exclude(Q(edu_id='') | Q(edu_id__isnull=True))

    def lookups(self, request, model_admin):
        """
        Список возможных фильтров
        """
        return self.LOOKUPS


class ConvertedProblemsFilter(admin.SimpleListFilter):
    """
    Фильтр конвертированных-неконвертированных задач
    """
    title = _(u'Конвертированность задач')
    parameter_name = 'converted'
    LOOKUPS = (
        ('1', _(u'Неконвертированные')),
        ('2', _(u'Конвертированные')),
    )

    def lookups(self, request, model_admin):
        """Поисковые параметры"""
        return self.LOOKUPS

    def queryset(self, request, queryset):
        """
        Ограничиваем конвертированными и неконвертированными по запросу.
        """
        if self.value() == self.LOOKUPS[0][0]:
            return queryset.filter(converted=False)
        elif self.value() == self.LOOKUPS[1][0]:
            return queryset.filter(converted=True)


class RawIdProblemFilter(admin.ListFilter):
    """Фильтр в виде формы для идентификатора вопроса"""
    title = _(u'Идентификатор задачи')
    parameter_name = 'problem_id'
    template = 'admin/problemhistory/raw_id_filter.html'

    def __init__(self, request, params, model, model_admin):
        super(RawIdProblemFilter, self).__init__(
            request, params, model, model_admin)
        self.value = None
        if self.parameter_name in params:
            self.value = params.pop(self.parameter_name)

    def expected_parameters(self):
        return [self.parameter_name]

    def has_output(self):
        return True

    def choices(self, cl):
        return [self.value] if self.value else []

    def queryset(self, request, queryset):
        try:
            value = int(self.value)
        except (TypeError, ValueError):
            return queryset
        return queryset.filter(problem_id=value)
