import json
from builtins import object

import six

from django import forms
from django.core.exceptions import ValidationError
from django.utils.translation import ugettext_lazy as _

from kelvin.common.admin import OwnerOnlyContentManagerModelFormMixin
from kelvin.problems.admin.widgets import MarkupWidget, ResourceDisplayWidget
from kelvin.problems.models import Problem, TextResource
from kelvin.problems.validators import RESOURCE_RE, ProblemValidators, TextResourceValidators


class MarkupCompositeField(forms.MultiValueField):
    """
    Поле для редактирования разметки задания в админке, разделает разметку на
    два поля: решение и остальное
    """
    widget = MarkupWidget

    def __init__(self, *args, **kwargs):
        """
        Задает поля для подполей: текстовые поля для разметки и решения и
        псевдополе для виджета отображения ресурсов
        """
        fields = (forms.CharField(widget=forms.Textarea,
                                  required=True),
                  forms.Field(widget=forms.HiddenInput,
                              required=False),
                  forms.CharField(widget=forms.Textarea,
                                  required=False,),
                  forms.CharField(widget=forms.Textarea,
                                  required=False,))
        super(MarkupCompositeField, self).__init__(
            *args, fields=fields, **kwargs)

    def compress(self, data_list):
        """
        Возвращает сжатое значение: к значению основного поля добавляется
        ключ `solution` из поля решения
        """
        data = data_list[0]
        if isinstance(data, six.string_types):
            try:
                # приводит значение к json
                data = json.loads(data)
            except ValueError as er:
                raise forms.ValidationError(u'Невалидный JSON: {0}'.format(
                    er.message))
        # добавляет по ключу `solution` значение из третьего подполя
        if data_list[2]:
            data['solution'] = data_list[2]
        # добавляет по ключу `public_solution` значение из четвертого подполя
        if data_list[3]:
            data['public_solution'] = data_list[3]
        return data


class ProblemAdminForm(forms.ModelForm):
    """
    Форма для админки задачи
    """
    markup = MarkupCompositeField(label=_(u'Разметка задачи'),
                                  require_all_fields=False)
    # чтобы форма сохраняла ресурсы
    resources = forms.Field(widget=forms.HiddenInput, required=False)

    class Meta(object):
        model = Problem
        exclude = (
            'cover',
        )

    def __init__(self, *args, **kwargs):
        """
        Перезаписывает начальные данные для `MarkupCompositeField`:
        в поле 'markup' лежит действительное значение markup задачи
        в поле 'resources' лежит маппинг ресурсов задачи и их url для отрисовки
        в `ResourceDisplayWidget`
        """
        super(ProblemAdminForm, self).__init__(*args, **kwargs)
        if kwargs.get('instance'):
            self.initial['markup'] = {
                'markup': kwargs['instance'].markup,
                'resources': [
                    (res.id, res.get_content_url())
                    for res in kwargs['instance'].resources.all()
                ],
            }

    def clean(self):
        """
        Проверяет поле разметки
        Изменения обычной разметки сохраняет в старую разметку, если они
        совпадали до изменения
        Перезаписывает поле ресурсов значениями, найденными в разметке для
        обычных задач
        """
        cleaned_data = super(ProblemAdminForm, self).clean()
        if self.errors.get('markup'):
            # если была обнаружена ошибка в поле, ничего не делаем
            return cleaned_data

        try:
            # TODO исправить
            ProblemValidators.validate_markup_json(
                cleaned_data.get('markup'))
            ProblemValidators.validate_markup_marker_types(
                cleaned_data.get('markup'))
            validated_resources = ProblemValidators.validate_markup(
                cleaned_data.get('markup'))
        except ValidationError as e:
            raise ValidationError({'markup': e.messages})

        if (
            self.instance.id and
            self.instance.markup == self.instance.old_markup and
            cleaned_data['old_markup'] == self.instance.markup
        ):
            cleaned_data['old_markup'] = cleaned_data['markup']
        cleaned_data['resources'] = validated_resources
        return cleaned_data


class TextResourceForm(OwnerOnlyContentManagerModelFormMixin, forms.ModelForm):
    """
    Кастомная форма для админки текстового ресурса
    """
    resource_preview = forms.Field(label=_(u'Превью ресурсов'),
                                   widget=ResourceDisplayWidget(),
                                   required=False)

    def __init__(self, *args, **kwargs):
        """
        Инициализирует значение для поля ресурсов
        """
        super(TextResourceForm, self).__init__(*args, **kwargs)
        if kwargs.get('instance'):
            self.initial['resource_preview'] = [
                (res.id, res.get_content_url())
                for res in kwargs['instance'].resources.all()
            ]

    def clean_formulas(self):
        """
        Проверяет, что формулы - это словарь/json-объект
        """
        if not isinstance(self.cleaned_data['formulas'], dict):
            raise ValidationError(u'Формулы должны быть словарем {}')
        return self.cleaned_data['formulas']

    def clean(self):
        """
        Проверяет, что все указанные тегами в поле `content`
        ресурсы существуют и привязывает их к текстовому ресурсу.
        """
        cleaned_data = super(TextResourceForm, self).clean()

        try:
            ids = TextResourceValidators.validate_content(
                cleaned_data['content'])
            if 'formulas' in cleaned_data:
                TextResourceValidators.validate_formula_set(
                    cleaned_data['content'], cleaned_data['formulas'])
        except ValidationError as e:
            self.add_error('content', e.message)
            return

        cleaned_data['resources'] = ids
        return cleaned_data
