import json
from builtins import object, str
from copy import deepcopy

from django.conf import settings
from django.conf.urls import url
from django.contrib import admin, messages
from django.contrib.auth import get_user_model
from django.core.exceptions import PermissionDenied
from django.core.urlresolvers import reverse
from django.db import transaction
from django.http.response import Http404
from django.shortcuts import redirect, render_to_response
from django.utils.translation import ugettext_lazy as _

from kelvin.common.admin import AvailableForSupportAdminMixin, UserBlameModelAdminMixin
from kelvin.courses.models import Course, CourseLessonLink
from kelvin.lessons.models import LessonProblemLink
from kelvin.problems.admin.converters import MarkupConverter, MarkupConverterError
from kelvin.problems.admin.filters import ConvertedProblemsFilter, ImportedProblemsFilter, RawIdProblemFilter
from kelvin.problems.admin.forms import ProblemAdminForm, TextResourceForm
from kelvin.problems.admin.utils import get_big_resource_preview, get_short_text
from kelvin.problems.models import Problem, ProblemHistory, TextResource, TextResourceContentType
from kelvin.result_stats.tasks import recalculate_problemstat
from kelvin.results.models import CourseLessonResult, LessonResult
from kelvin.results.utils import recalculate_clesson_result, recalculate_lesson_result

User = get_user_model()


class ProblemAdmin(UserBlameModelAdminMixin, AvailableForSupportAdminMixin, admin.ModelAdmin):
    """
    Админка для задачи
    """
    change_form_template = 'admin/problems/problem_change_form.html'
    list_display = (
        'id',
        'get_front_link',
        'get_big_resource_preview',
        'get_short_text',
        'name',
        'get_solution',
        'get_screenshot',
        'custom_answer',
    )
    search_fields = ('id', 'name', 'markup')
    list_filter = (
        ImportedProblemsFilter,
        ConvertedProblemsFilter,
        'subject',
        'custom_answer',
    )

    readonly_fields = (
        'get_front_link',
        'get_problemstat',
        'get_versions_link',
        'created_by',
        'modified_by',
    )
    raw_id_fields = (
        'meta',
        'tips',
    )
    actions = ('calculate_problem_stat', )

    form = ProblemAdminForm

    class Media(object):
        css = {
            'all': ('css/problem-list.css',)
        }

    def get_queryset(self, request):
        """
        Добавляет ресурсы, статистики и занятия
        """
        qs = super(ProblemAdmin, self).get_queryset(request)
        return qs.select_related('problemstat').prefetch_related(
            'resources',
            'lesson_set',
        )

    def save_model(self, request, obj, form, change):
        """
        Создает версию задачи
        """
        super(ProblemAdmin, self).save_model(request, obj, form, change)
        ProblemHistory.add_problem_version(
            obj, request.user, u'Изменения в админке')

    def render_change_form(self, request, context, **kwargs):
        """
        Добавляет в контекст рендеринга список курсов, в которых используется
        задача
        """
        problem = context.get('original')
        if problem:
            problem_courses = (
                Course.objects.filter(lessons__problems=problem.id)
                .only('id', 'name')
                .distinct()
            )
            message = (
                u'Основная и старая разметки совпадают. Если нужны '
                u'изменения - достаточно внести их в основную'
            ) if problem.markup == problem.old_markup else (
                u'Основная и старая разметки НЕ совпадают. Если нужны '
                u'изменения - внесите их в обе разметки'
            )
            messages.add_message(request, messages.WARNING, message)
            context.update({'problem_courses': problem_courses})
        return super(ProblemAdmin, self).render_change_form(request, context,
                                                            **kwargs)

    def formfield_for_foreignkey(self, db_field, request=None, **kwargs):
        """Проставляет дефолтное значение для владельца"""
        if (db_field.name == 'owner' and request and
                request.user.is_content_manager):
            kwargs['initial'] = request.user.id
        return super(ProblemAdmin, self).formfield_for_foreignkey(
            db_field, request, **kwargs)

    def get_urls(self):
        """
        Добавляет адрес для конвертации и копирования задачи
        """
        info = self.model._meta.app_label, self.model._meta.model_name
        convert_url = url(
            r'^(.+)/convert/$',
            self.admin_site.admin_view(self.convert_view),
            name='{0}_{1}_convert'.format(*info),
        )
        copy_url = url(
            r'^(.+)/copy/$',
            self.admin_site.admin_view(self.copy_view),
            name='{0}_{1}_copy'.format(*info),
        )
        recalculate_url = url(
            r'^(.+)/recalculate/$',
            self.admin_site.admin_view(self.recalculate_view),
            name='{0}_{1}_recalculate'.format(*info),
        )

        return [convert_url, copy_url, recalculate_url] + super(
            ProblemAdmin, self).get_urls()

    def get_field_queryset(self, db, db_field, request):
        """
        Для поле `owner` возвращает только контент-менеджеров
        """
        if db_field.name == 'owner':
            return User.objects.filter(is_content_manager=True)

        return super(ProblemAdmin, self).get_field_queryset(
            db, db_field, request)

    def get_short_text(self, obj):
        """
        Возвращает содержимого первого контейрена из разметки задачи
        """
        return get_short_text(obj)
    get_short_text.short_description = _(u'Текст')

    def get_front_link(self, obj):
        """
        Выводит ссылку на фронт
        """
        return (u'<a href="{0}lab/problems/{1}/'
                u'?show_solution=1&show_answers=1">Посмотреть</a>'.format(
                    settings.FRONTEND_HOST, obj.id,))
    get_front_link.short_description = _(u'Ссылка на фронт')
    get_front_link.allow_tags = True

    def get_problemstat(self, problem):
        """
        Ссылка на статистику (если есть)
        """
        if hasattr(problem, 'problemstat'):
            link = reverse('admin:result_stats_problemstat_change',
                           args=(problem.problemstat.id, ))
            return u'<a href="{0}" target="_blank">Посмотреть</a>'.format(link)
        else:
            link = reverse('admin:result_stats_problemstat_create',
                           args=(problem.id, ))
            return u'<a href="{0}" target="_blank">Создать</a>'.format(link)
    get_problemstat.short_description = _(u'Статистика')
    get_problemstat.allow_tags = True

    def get_screenshot(self, obj):
        """
        Выводит скриншот ввиде img тэга
        """
        if not obj.screenshot:
            return ''
        return (u'<img src="{0}" height="50px" />'
                .format(obj.screenshot.url)
                if obj.screenshot else '')
    get_screenshot.short_description = _(u'Скриншот')
    get_screenshot.allow_tags = True

    def get_solution(self, obj):
        """
        Выводит, есть ли решение у задачи
        """
        return bool(obj.markup.get('solution', ''))
    get_solution.boolean = True
    get_solution.short_description = _(u'Решение')

    def get_big_resource_preview(self, obj):
        """Выводит превью наибольшего ресурса"""
        return get_big_resource_preview(obj)
    get_big_resource_preview.short_description = u'Ресурс из задачи'
    get_big_resource_preview.allow_tags = True

    def get_versions_link(self, obj):
        """Ссылка на версии задачи"""
        return (
            u'<a href="{0}?problem_id={1}" target="_blank">Ссылка</a>'
            .format(reverse('admin:problems_problemhistory_changelist'),
                    obj.id)
        )
    get_versions_link.short_description = _(u'Версии задачи')
    get_versions_link.allow_tags = True

    def calculate_problem_stat(self, request, queryset):
        """
        Запускает подсчет статистики для каждой задачи в отдельном селери-таске
        """
        for problem in queryset:
            recalculate_problemstat.delay(problem.id)
    calculate_problem_stat.short_description = u'Пересчитать статистику'

    def convert_view(self, request, pk):
        """
        Конвертация задачи: делает копию `old_markup`, конвертирует ее,
        записывает в `markup` и выставляет флаг конвертированности
        """
        if request.method != 'POST':
            raise Http404
        try:
            problem = Problem.objects.get(pk=pk)
        except (ValueError, TypeError, Problem.DoesNotExist):
            raise Http404

        if not self.has_change_permission(request, problem):
            raise PermissionDenied
        width_1 = 'width' in request.GET
        try:
            problem.markup = MarkupConverter(width_1=width_1).convert(
                deepcopy(problem.old_markup))
        except MarkupConverterError as e:
            messages.add_message(
                request,
                messages.ERROR,
                u'Ошибка при конвертации: {0}'.format(e.message),
            )
        else:
            problem.converted = True
            problem.save()
            ProblemHistory.add_problem_version(
                problem, request.user, message=u'Конвертация задачи')
            messages.add_message(request, messages.INFO,
                                 u'Задача успешно конвертирована')

        return redirect('admin:problems_problem_change', problem.id)

    def copy_view(self, request, pk):
        """
        Копирует задачу с изменением названия и автора
        Создает версию новой задачи
        """
        if request.method != 'POST':
            raise Http404
        try:
            problem = Problem.objects.get(pk=pk)
        except (ValueError, TypeError, Problem.DoesNotExist):
            raise Http404

        if not self.has_change_permission(request, problem):
            raise PermissionDenied

        with transaction.atomic():
            resources = list(problem.resources.all())
            tips = list(problem.tips.all())
            problem.name = u'Копия - {0}'.format(problem.id)
            problem.pk = None
            problem.owner = request.user
            problem.save()
            problem.resources = resources
            problem.tips = tips
            ProblemHistory.add_problem_version(
                problem, request.user, u'Копирование задачи')

        messages.add_message(request, messages.INFO,
                             u'Задача успешно создана')

        return redirect('admin:problems_problem_change', problem.id)

    def recalculate_view(self, request, pk):
        """
        Пересчитывает результаты по задаче
        """
        lesson_ids = (
            LessonProblemLink.objects
            .filter(problem_id=pk)
            .values_list('lesson_id', flat=True)
        )
        clesson_ids = (
            CourseLessonLink.objects
            .filter(lesson_id__in=lesson_ids)
            .values_list('id', flat=True)
        )
        for result in CourseLessonResult.objects.filter(
                summary__clesson_id__in=clesson_ids):
            recalculate_clesson_result(result)

        for result in LessonResult.objects.filter(
                summary__lesson_id__in=lesson_ids):
            recalculate_lesson_result(result)

        info = self.model._meta.app_label, self.model._meta.model_name
        return redirect('admin:{0}_{1}_change'.format(*info), pk)


class TextResourceAdmin(admin.ModelAdmin):
    """
    Админка для текстовых ресурсов (шпаргалок, etc.)
    """
    form = TextResourceForm

    change_form_template = 'admin/problems/change_form_with_mathjax.html'
    list_display = (
        'name',
        'content_type_object',
        'get_front_link',
    )
    readonly_fields = (
        'get_front_link',
    )
    fields = (
        'name',
        'owner',
        'content_type_object',
        'content',
        'resource_preview',
        'formulas',
        'subject',
        'themes',
        'resources',
        'get_front_link',
    )
    list_filter = (
        'content_type_object',
        'themes',
    )
    filter_horizontal = (
        'themes',
    )
    raw_id_fields = (
        'owner',
        'resources',
    )
    search_fields = ('name', 'content')

    def get_queryset(self, request):
        """
        Дополнительно подтягивает связи с занятиями и сами занятия
        """
        return super(TextResourceAdmin, self).get_queryset(
            request).prefetch_related('lessonproblemlink_set',
                                      'lessonproblemlink_set__lesson')

    def formfield_for_manytomany(self, db_field, request=None, **kwargs):
        """
        Отображаем выбор тем с отступами, соответствующими глубине
        """
        field = super(TextResourceAdmin, self).formfield_for_manytomany(
            db_field, request, **kwargs)
        if db_field.name == 'themes':
            queryset = field.choices.queryset
            field.choices = []
            for theme in queryset:
                field.choices.append((
                    theme.id,
                    u"{indent} {code}: {theme}".format(
                        code=theme.code,
                        indent="--" * theme.level,
                        theme=theme,
                    ),
                ))

        return field

    def get_front_link(self, obj):
        """
        Выводит ссылку на фронт
        """
        return (u'<a href="{0}lab/theories/{1}/">Посмотреть</a>'.format(
            settings.FRONTEND_HOST, obj.id,))
    get_front_link.short_description = _(u'Ссылка на фронт')
    get_front_link.allow_tags = True


class TextResourceContentTypeAdmin(admin.ModelAdmin):
    """
    Админка для типов содержимого.
    """
    list_display = (
        'name',
        'get_resource_link',
    )
    raw_id_fields = (
        'resource',
    )

    def get_queryset(self, request):
        """
        Подтягиваем дополнительно ресурс для типа контента
        """
        return (
            super(TextResourceContentTypeAdmin, self)
            .get_queryset(request)
            .select_related('resource')
        )

    def get_resource_link(self, obj):
        """
        Выводит ссылку на файл ресурса

        :type obj: TextResourceContentType
        """
        return u'<a href="{0}">{0}</a>'.format(obj.resource.get_content_url())
    get_resource_link.short_description = _(u'Ссылка на картинку')
    get_resource_link.allow_tags = True


class ProblemHistoryAdmin(admin.ModelAdmin):
    """
    Админка версий задач
    """
    change_form_template = 'admin/problemhistory/change_form.html'
    compare_versions_template = 'admin/problemhistory/compare_versions.html'

    list_display = (
        'id',
        'problem',
        'author',
        'date_created',
        'message',
    )
    list_filter = (
        RawIdProblemFilter,
    )
    raw_id_fields = (
        'problem',
        'author',
    )
    actions = (
        'compare_versions_action',
    )

    readonly_fields = (
        'get_problem_link',
        'get_versions_link',
        'get_comparison_link',
        'author',
        'message',
        'get_markup',
        'get_old_markup',
    )
    exclude = (
        'problem',
        'markup',
        'old_markup',
    )

    def has_add_permission(self, request):
        return False

    def has_delete_permission(self, request, obj=None):
        return False

    def get_queryset(self, request):
        """
        Добавляет автора в запрос
        """
        qs = super(ProblemHistoryAdmin, self).get_queryset(request)
        return qs.select_related('author', 'problem')

    def get_urls(self):
        """
        Добавляет адрес для страницы сравнения версий
        """
        info = self.model._meta.app_label, self.model._meta.model_name
        compare_url = url(
            r'^compare/(?P<first>\d+)/(?P<second>\d+)/$',
            self.admin_site.admin_view(self.compare_versions_view),
            name='{0}_{1}_compare'.format(*info),
        )
        return [compare_url] + super(ProblemHistoryAdmin, self).get_urls()

    def get_problem_link(self, obj):
        """Ссылка на задачу"""
        return (
            u'<a href="{0}" target="_blank">{1}</a>'
            .format(reverse('admin:problems_problem_change',
                            args=(obj.problem_id,)),
                    str(obj.problem))
        )
    get_problem_link.short_description = _(u'Задачи')
    get_problem_link.allow_tags = True

    def get_versions_link(self, obj):
        """Ссылка на версии задачи"""
        return (
            u'<a href="{0}?problem_id={1}" target="_blank">Ссылка</a>'
            .format(reverse('admin:problems_problemhistory_changelist'),
                    obj.problem_id)
        )
    get_versions_link.short_description = _(u'Версии задачи')
    get_versions_link.allow_tags = True

    def get_comparison_link(self, obj):
        """Ссылка на предыдующую версию задачи"""
        prev = (
            ProblemHistory.objects
            .filter(problem=obj.problem, date_created__lt=obj.date_created)
            .order_by('-date_created')
            .first()
        )
        if not prev:
            return u'Нет прошлых версий'
        return (
            u'<a href="{0}" target="_blank">Ссылка</a>'
            .format(reverse('admin:problems_problemhistory_compare',
                            kwargs={'first': prev.id, 'second': obj.id}))
        )
    get_comparison_link.short_description = _(
        u'Сравнение с предыдущей версией'
    )
    get_comparison_link.allow_tags = True

    def get_markup(self, obj):
        return u'<pre>{0}</pre>'.format(
            json.dumps(obj.markup, indent=2, ensure_ascii=False))
    get_markup.short_description = _(u'Разметка задачи (markup)')
    get_markup.allow_tags = True

    def get_old_markup(self, obj):
        if obj.old_markup:
            return u'<pre>{0}</pre>'.format(
                json.dumps(obj.old_markup, indent=2, ensure_ascii=False))
        return ''
    get_old_markup.short_description = _(
        u'Устаревшая разметка задачи (old_markup)'
    )
    get_old_markup.allow_tags = True

    def compare_versions_action(self, request, queryset):
        """
        Редиректит на сравнение 2 версий
        """
        if len(queryset) != 2:
            message = (u'Надо выбрать 2 версии для сравнения, выбрано {0}'
                       .format(len(queryset)))
            messages.add_message(request, messages.WARNING, message)
            return
        info = self.model._meta.app_label, self.model._meta.model_name
        first, second = sorted(queryset, key=lambda obj: obj.date_created)
        return redirect('admin:{0}_{1}_compare'.format(*info),
                        first=first.id, second=second.id)
    compare_versions_action.short_description = u'Сравнить версии'

    def compare_versions_view(self, request, first, second):
        """Сравнение двух версий задач"""
        objects = {
            str(obj.id): obj
            for obj in ProblemHistory.objects.filter(id__in=[first, second])
        }
        first = objects[first]
        second = objects[second]
        opts = ProblemHistory._meta
        info = self.model._meta.app_label, self.model._meta.model_name
        return render_to_response(
            self.compare_versions_template,
            context={
                'opts': opts,
                'app_label': opts.app_label,
                'first': {
                    'name': str(first),
                    'url': reverse('admin:{0}_{1}_change'.format(*info),
                                   args=(first.id,)),
                    'markup': json.dumps(first.markup, ensure_ascii=False,
                                         indent=2).replace('`', '&#96;'),
                    'old_markup': (json.dumps(first.old_markup,
                                              ensure_ascii=False, indent=2)
                                   .replace('`', '&#96;')),
                },
                'second': {
                    'name': str(second),
                    'url': reverse('admin:{0}_{1}_change'.format(*info),
                                   args=(second.id,)),
                    'markup': json.dumps(second.markup, ensure_ascii=False,
                                         indent=2).replace('`', '&#96;'),
                    'old_markup': (json.dumps(second.old_markup,
                                              ensure_ascii=False, indent=2)
                                   .replace('`', '&#96;')),
                },
            },
        )


admin.site.register(Problem, ProblemAdmin)
admin.site.register(TextResource, TextResourceAdmin)
admin.site.register(TextResourceContentType, TextResourceContentTypeAdmin)
admin.site.register(ProblemHistory, ProblemHistoryAdmin)
