import json
from builtins import object

from django import forms
from django.conf import settings
from django.core.urlresolvers import reverse
from django.forms import widgets
from django.template.loader import render_to_string
from django.utils.safestring import mark_safe

from kelvin.common.admin import JsonValidatingTextarea


class UploadWidget(widgets.TextInput):
    """
    From http://diy-django.readthedocs.org/en/latest/ajaxupload.html
    """

    def render(self, name, value, attrs=None, renderer=None):
        """
        Возвращает HTML разметку виджета
        :param name: Имя виджета
        :param value: Текущие данные виджета
        :param attrs: Дополнительные атрибуты
        :param renderer: Рендерер, если когда-то передем на 2 версию,
            станет обязательным.
        :return: HTML разметка виджета
        """
        attrs = attrs or {}
        attrs['class'] = 'vTextField'
        input = super(UploadWidget, self).render(name, value, attrs=attrs)
        value = '<a>None</a>'

        if settings.DEFAULT_FILE_STORAGE == (
                'storages.backends.s3boto.S3BotoStorage'
        ):
            href = 'https://{0}.s3.amazonaws.com/'.format(
                settings.AWS_STORAGE_BUCKET_NAME,
            )
        else:
            href = '/media/'
        onchange = r'''
f = this.files[0]
this.value = ''
var xhr = new XMLHttpRequest()
xhr.open('POST', '/simple_upload/' + f.name, true)
xhr.setRequestHeader('X-CSRFToken', this.form.csrfmiddlewaretoken.value)
xhr.upload.status = this.parentNode.firstChild.nextElementSibling // <span>
xhr.upload.status.innerHTML = 'pending...'
xhr.upload.onprogress = function(e){ this.status.innerHTML = Math.round(100 * e.loaded / e.total) + 'percent uploaded' }
xhr.send(f)
xhr.onload = function(){
  this.upload.status.innerHTML = '<a href=\'%s' + encodeURI(this.responseText) + '\' target=_blank>view</a>'
  this.upload.status.previousElementSibling.value = this.responseText // <input type=hidden>
}''' % href  # noqa
        assert '"' not in onchange
        return mark_safe(
            '<p>{0}<span>{1}</span><br><input type="file" onchange="{2}" '
            'value="upload"></p>'.format(input, value, onchange))


class ResourceDisplayWidget(widgets.Widget):
    """
    Простой виджет для отображения списка ресурсов
    """
    RESOURCE_CHANGE_REVERSE = 'admin:resources_resource_change'
    RESOURCE_DISPLAY_TEMPLATE = 'admin/problems/resource_display_widget.html'

    template_name = 'admin/problems/resource_display_widget.html'

    class Media(object):
        css = {
            'all': ('css/resourcedisplay.css', )
        }

    def render(self, name, value, attrs=None, renderer=None):
        """
        Возвращает HTML разметку виджета
        :param name: Имя виджета
        :param value: Текущие данные виджета
        :param attrs: Дополнительные атрибуты
        :param renderer: Рендерер, если когда-то передем на 2 версию,
            станет обязательным.
        :return: HTML разметка виджета
        """
        if isinstance(value, str):
            value = json.loads(value)
        if value:
            context = {
                'res_data': [{
                    'id': res_id,
                    'file_url': res_url,
                    'admin_change_url': reverse(
                        self.RESOURCE_CHANGE_REVERSE, args=(res_id, )),
                } for res_id, res_url in value]
            }
            return render_to_string(
                self.RESOURCE_DISPLAY_TEMPLATE,
                context=context,
            )
        else:
            return u''


class MarkupWidget(forms.MultiWidget):
    """
    Виджет для поля разметки задачи.
    Содержит в следующей последовательности: поле для ввода основной части
    разметки, превью ресурсов в разметке, поле для ввода решения, поле
    для простой загрузки файла, не привязанного к модели
    """
    FORMAT_TEMPLATE = 'admin/problems/markup_widget_format.html'

    def __init__(self, attrs=None):
        """
        Инициализирует виджет с нужными виджетами подполей
        """
        text_widget_attrs = {'cols': 40, 'class': "vLargeTextField"}
        # параметры для представления json строкой
        self.dump_kwargs = {
            'cls': json.JSONEncoder,
            'separators': (',', ':'),
            "indent": 2,
            "ensure_ascii": False,
        }
        super(MarkupWidget, self).__init__(
            widgets=(JsonValidatingTextarea(attrs=text_widget_attrs),
                     ResourceDisplayWidget(),
                     forms.Textarea(attrs=text_widget_attrs),
                     forms.Textarea(attrs=text_widget_attrs),
                     UploadWidget()),
            attrs=attrs)

    def decompress(self, value):
        """
        Разделяет начальное значение на основную разметку, список ресурсов
        (если есть) и решения
        """
        if value is None:
            return None, [], None
        # выносим значение поля `solution, public_solution` в отдельное поле
        markup = value['markup']
        solution = markup.pop('solution', None)
        public_solution = markup.pop('public_solution', None)
        return (
            json.dumps(markup, **self.dump_kwargs),
            json.dumps(value['resources']),
            solution,
            public_solution,
        )

    def render(self, name, value, attrs=None, renderer=None):
        """
        Возвращает HTML разметку виджета: поле для ввода основной части
        разметки, превью ресурсов в разметке, поле для ввода решения, поле
        для простой загрузки файла, не привязанного к модели
        :param name: Имя виджета
        :param value: Текущие данные виджета
        :param attrs: Дополнительные атрибуты
        :param renderer: Рендерер, если когда-то передем на 2 версию,
            станет обязательным.
        :return: HTML разметка виджета
        """
        context = self.get_context(name, value, attrs)

        fields = [
            'main_markup',
            'resource_display',
            'solution',
            'public_solution',
            'upload',
        ]
        final_context = {}
        for i, key in enumerate(fields):
            widget_context = context['widget']['subwidgets'][i]
            final_context[key] = self.widgets[i].render(
                widget_context['name'],
                widget_context['value'],
                widget_context['attrs'],
            )

        return render_to_string(
            self.FORMAT_TEMPLATE,
            context=final_context,
        )
