import logging
from builtins import object, str
from copy import deepcopy

from past.utils import old_div

from kelvin.problems.markers import Marker
from kelvin.problems.utils import get_markers_dict_from_layout

logger = logging.getLogger(__name__)


class Answer(object):
    """
    Ответ на вопрос
    """
    # статусы ответов
    CORRECT = Marker.CORRECT
    INCORRECT = Marker.INCORRECT
    UNCHECKED = Marker.UNCHECKED

    # статусы сводных результатов
    SUMMARY_INCORRECT = 0
    SUMMARY_CORRECT = 1

    # тип выставления баллов за задачу
    ATTEMPTS_COUNT_TYPE = 1
    MISTAKES_COUNT_TYPE = 2

    def __init__(self, markers, theory=None, custom_answer=None, completed=True,
                 spent_time=None, mistakes=None, max_mistakes=None,
                 points=None, comment='', checked_points=None, answered=False):
        """
        Задаем поля объекта

        :param markers: ответ на вопрос по маркерам
        :param theory: объект теоркии, которую прльзоватаель просмотрел
        :param custom_answer: поле ручной проверки
        :param completed: завершенный ответ
        :param spent_time: число секунд, потраченное на ответ
        :param mistakes: количество ошибок в ответе
        :param max_mistakes: максимальное количество ошибок в вопросе
        :param points: число баллов за попытку
        :param checked_points: число баллов за попытку после ручной проверки
        :param comment: комментарий проверяющего к попытке
        :param answered: нажималась ли кнопка ответа, по умолчанию False
        """
        self.markers = markers
        self.theory = theory
        self.custom_answer = custom_answer
        self.mistakes = mistakes
        self.max_mistakes = max_mistakes
        self.spent_time = spent_time
        self.completed = completed
        self.points = points
        self.comment = comment
        self.checked_points = checked_points
        self.answered = answered

    @classmethod
    def get_summary(cls, answers, max_attempts, hidden_results=False,
                    force_results=False, max_points=None):
        """
        Вернуть сводку попыток по вопросу
        """
        if not answers:
            return {
                'status': cls.SUMMARY_INCORRECT if force_results
                else None,
                'answered': force_results
            }
        attempt_number = len(answers)
        time = sum([answer.get('spent_time') or 0 for answer in answers])
        last_attempt = answers[-1]
        answered = last_attempt.get('answered', False)
        points = last_attempt.get('points', 0)

        if hidden_results:
            status = None
        elif ('custom_answer' in last_attempt and
                type(last_attempt['custom_answer']) is list and
                len(last_attempt['custom_answer'])):
            # Задача с ручной проверкой
            status = None
            last_custom_answer = last_attempt['custom_answer'][-1]
            if 'points' in last_custom_answer:
                points = last_custom_answer['points']
                if 'status' in last_custom_answer:
                    status = last_custom_answer['status']
        else:
            if not last_attempt['completed']:
                status = None
                attempt_number -= 1
            elif last_attempt['mistakes'] is None:
                status = None
            elif last_attempt['mistakes'] == 0:
                status = cls.SUMMARY_CORRECT
            else:
                status = cls.SUMMARY_INCORRECT

            if force_results and status != cls.SUMMARY_CORRECT:
                status = cls.SUMMARY_INCORRECT

        return {
            'answered': answered or status is not None,
            'attempt_number': attempt_number,
            'time': time,
            'status': status,
            'max_attempts': max_attempts,
            'points': points,
        }

    @staticmethod
    def get_points_for_custom_answer(answers):
        """
        Возвращает баллы за ручную проверку
        """
        if not answers:
            return 0

        # Проверяем, что есть объект ручной проверки
        if not any(answer.get('custom_answer') for answer in answers):
            return 0

        latest_answer = answers[-1]

        latest_answer['points'] = None
        custom_answers = latest_answer.get('custom_answer')

        # len(custom_answer) == 0 здесь учтен
        if not custom_answers:
            return None

        latest_custom_answer = custom_answers[-1]
        if latest_custom_answer['type'] != CustomAnswer.Type.CHECK:
            return None

        # Если есть поле с ручной проверкой, то нужно обновить баллы
        latest_answer['points'] = latest_custom_answer['points']
        return latest_answer['points']

    @classmethod
    def get_points(cls, answers, max_attempts, max_points, count_type):
        """
        Вернуть баллы по вопросу.
        Изменяем последнюю попытку, записывая туда число баллов, если баллы
        не были посчитаны ранее.

        Подсчет по попыткам (`count_type==ATTEMPTS_COUNT_TYPE`) состоит в
        следующем.
        У каждой задачи есть предустановленное количество попыток N,
        больше которого ученик не может сделать в рамках занятия,
        а также сложность m.
        За каждую попытку, сделанную учеником, из максимального числа баллов
        за задачу вычитается число m/N.

        Подсчет по баллам (`count_type==MISTAKES_COUNT_TYPE`) за каждую
        ошибку вычитает балл; если допущено максимальное количество ошибок,
        то выставляется 0 баллов.

        Параметр :max_points: может быть словарем, в котором указано, за какой
        маркер какой максимальный балл можно получить.
        """
        if not answers:
            return 0

        points = 0
        if answers[-1]['mistakes'] is None:
            # вопрос содержит непроверяемый маркер
            points = None
        else:
            if isinstance(max_points, dict):
                # проставляем баллы по маркерам
                if count_type == cls.ATTEMPTS_COUNT_TYPE:
                    # подсчет по числу попыток
                    for marker_id, marker_answer in (
                        iter(answers[-1]['markers'].items())
                    ):
                        if (
                            marker_answer['mistakes'] == 0 and
                            marker_id in max_points
                        ):
                            marker_max_points = max_points[marker_id]
                            attempt_penalty = (
                                old_div(marker_max_points, float(max_attempts))
                            )
                            failed_attempts = len(answers) - 1
                            points += max_points[marker_id]
                else:
                    # подсчет по числу ошибок
                    for marker_id, marker_answer in (
                        iter(answers[-1]['markers'].items())
                    ):
                        if (
                            marker_answer['mistakes'] !=
                            marker_answer['max_mistakes'] and
                            marker_id in max_points
                        ):
                            points += max_points[marker_id]
            else:
                # проставляем баллы целиком за задачу
                if count_type == cls.ATTEMPTS_COUNT_TYPE:
                    # подсчет по числу попыток
                    if answers[-1]['mistakes'] == 0:
                        attempt_penalty = old_div(max_points, float(max_attempts))
                        failed_attempts = len(answers) - 1
                        points = max_points
                else:
                    # подсчет по числу ошибок
                    if answers[-1]['mistakes'] != answers[-1]['max_mistakes']:
                        # points = max(max_points - answers[-1]['mistakes'], 0)
                        points = max_points

        answers[-1]['points'] = points  # TODO: EDU-274
        if answers[-1].get('checked_points') is not None:
            return answers[-1]['checked_points']
        return points


class CustomAnswer(object):
    """
    Модель сообщения для ручной проверки
    """
    AVAILABLE_STATUSES_FOR_CHECK = (
        Answer.SUMMARY_CORRECT, Answer.SUMMARY_INCORRECT
    )

    # Типы сообщений
    class Type(object):
        SOLUTION = 'solution'
        CHECK = 'check'

        AVAILABLE_TYPES = (SOLUTION, CHECK)

        # Типы сообщений, которые могут выставлять оценки
        TYPES_WITH_POINTS = (CHECK, )

        # Типы сообщений, которые разрешено отправлять после окончания
        # контрольной
        TYPES_THAT_MAY_BY_AFTER_CONTROL_WORK = (CHECK, )

        # Типы сообщений, которые могут появиться в завершенной попытке
        TYPES_THAT_MAY_BE_IN_COMPLETED_ATTEMPTS = (CHECK, )


def check_answer(problem, answer):
    """
    Проверить ответ на вопрос

    Ответ считается правильным, если все ответы на маркеры правильные

    :param problem: вопрос, на который дан ответ
    :type problem: Problem
    :param answer: ответ пользователя, возможно уже проверенный
    :type answer: Answer
    :return: ответ с проставленными статусами
    :rtype: Answer
    """
    if answer.mistakes is not None and answer.max_mistakes is not None:
        # ответ уже проверен
        return answer

    answer_markers = deepcopy(answer.markers) or {}
    problem_markers = get_markers_dict_from_layout(problem.markup['layout'])
    answer_mistakes = 0
    max_answer_mistakes = 0

    # для каждого маркера делаем проверку, модифицируя `answer_markers`
    for id_ in problem_markers.keys():
        if answer_markers is None:
            logger.warning('Skipping marker %s for problem %s because '
                           'answer_markers are None', id_, problem.pk)
            continue

        if id_ not in answer_markers:
            # в ответе пишем про все маркеры из вопроса
            answer_markers[id_] = {}
        item = answer_markers[id_]
        if 'mistakes' not in item or 'max_mistakes' not in item:
            markup = problem.markup
            marker = Marker(
                problem_markers[id_],
                markup['answers'][str(id_)],
                markup['checks'].get(str(id_)),
            )
            answer_status, mistakes = marker.check(item.get('user_answer'))
            item['answer_status'] = answer_status
            item['mistakes'] = mistakes
            item['max_mistakes'] = marker.max_mistakes
        if item['mistakes'] is None:
            # маркер непроверяемый
            answer_mistakes = None
        if answer_mistakes is not None:
            answer_mistakes += item['mistakes']
        max_answer_mistakes += item['max_mistakes']

    return Answer(markers=answer_markers,
                  # theory={},
                  custom_answer=answer.custom_answer,
                  completed=answer.completed,
                  spent_time=answer.spent_time,
                  mistakes=answer_mistakes,
                  max_mistakes=max_answer_mistakes,
                  comment=answer.comment,
                  checked_points=answer.checked_points,
                  answered=answer.answered)
