import operator
import re
from builtins import object, str
from collections import Counter
from copy import deepcopy
from fractions import Fraction
from functools import reduce

from past.utils import old_div

from kelvin.common.algorithms import kuhn
from kelvin.common.utils import OrderedCounter, generate_graph_from_counters

RE_DECIMAL_FRACTION = re.compile(
    r'^\s* (0 | (-? 0(\.|\,)\d+) | (-?[1-9]\d* ((\.|\,)\d+)?)) \s*$',
    flags=re.X,
)
RE_MIXED_FRACTION = re.compile(
    r'^\s* (0 | (-?(([1-9]\d*)\s+)? ([1-9]\d*) \s*/\s* [1-9]\d*)) \s*$',
    flags=re.X,
)


class CommonChecker(object):
    """
    Общие функции и определения проверок
    """
    COMMON_DEFINITIONS = {
        'BOOL_VALUE_OPERATORS': {
            'oneOf': [
                {'$ref': '#/check_definitions/IS_NONE'},
                {'$ref': '#/check_definitions/EQUAL'},
                {'$ref': '#/check_definitions/NOT'},
                {'$ref': '#/check_definitions/OR'},
                {'$ref': '#/check_definitions/AND'},
                {'$ref': '#/check_definitions/DIFFERENT'},
                {'$ref': '#/check_definitions/MORE'},
                {'$ref': '#/check_definitions/EQMORE'},
                {'$ref': '#/check_definitions/LESS'},
                {'$ref': '#/check_definitions/EQLESS'},
                {'$ref': '#/check_definitions/IS_SUBSET'},
                {'$ref': '#/check_definitions/UNIQUE_ITEMS'},
                {'$ref': '#/check_definitions/IS_PERMUTATION_FROM'},
            ],
        },
        'EXPRESSION': {
            'oneOf': [
                # булевы операторы
                {'$ref': '#/check_definitions/IS_NONE'},
                {'$ref': '#/check_definitions/EQUAL'},
                {'$ref': '#/check_definitions/NOT'},
                {'$ref': '#/check_definitions/OR'},
                {'$ref': '#/check_definitions/AND'},
                {'$ref': '#/check_definitions/DIFFERENT'},
                {'$ref': '#/check_definitions/MORE'},
                {'$ref': '#/check_definitions/EQMORE'},
                {'$ref': '#/check_definitions/LESS'},
                {'$ref': '#/check_definitions/EQLESS'},
                {'$ref': '#/check_definitions/IS_SUBSET'},
                {'$ref': '#/check_definitions/UNIQUE_ITEMS'},
                {'$ref': '#/check_definitions/IS_PERMUTATION_FROM'},

                # остальные
                {'$ref': '#/check_definitions/NONE'},
                {'$ref': '#/check_definitions/SIZE'},
                {'$ref': '#/check_definitions/NUMBER'},
                {'$ref': '#/check_definitions/STRING'},
                {'$ref': '#/check_definitions/BOOLEAN'},
                {'$ref': '#/check_definitions/SUM'},
                {'$ref': '#/check_definitions/MINUS'},
                {'$ref': '#/check_definitions/MULT'},
                {'$ref': '#/check_definitions/REMAINDER'},
                {'$ref': '#/check_definitions/QUOTIENT'},
            ],
        },
        'OPERATORS': {
            'oneOf': [
                # булевы операторы
                {'$ref': '#/check_definitions/IS_NONE'},
                {'$ref': '#/check_definitions/EQUAL'},
                {'$ref': '#/check_definitions/NOT'},
                {'$ref': '#/check_definitions/OR'},
                {'$ref': '#/check_definitions/AND'},
                {'$ref': '#/check_definitions/DIFFERENT'},
                {'$ref': '#/check_definitions/MORE'},
                {'$ref': '#/check_definitions/EQMORE'},
                {'$ref': '#/check_definitions/LESS'},
                {'$ref': '#/check_definitions/EQLESS'},
                {'$ref': '#/check_definitions/IS_SUBSET'},
                {'$ref': '#/check_definitions/UNIQUE_ITEMS'},
                {'$ref': '#/check_definitions/IS_PERMUTATION_FROM'},

                # остальные
                {'$ref': '#/check_definitions/NONE'},
                {'$ref': '#/check_definitions/NUMBER'},
                {'$ref': '#/check_definitions/RATIONAL'},
                {'$ref': '#/check_definitions/SIZE'},
                {'$ref': '#/check_definitions/STRING'},
                {'$ref': '#/check_definitions/BOOLEAN'},
                # {'$ref': '#/check_definitions/INPUT'},
                {'$ref': '#/check_definitions/SUM'},
                {'$ref': '#/check_definitions/MINUS'},
                {'$ref': '#/check_definitions/MULT'},
                {'$ref': '#/check_definitions/REMAINDER'},
                {'$ref': '#/check_definitions/QUOTIENT'},

                # множества
                {'$ref': '#/check_definitions/UNION'},
                {'$ref': '#/check_definitions/MULTIUNION'},
            ],
        },
        'IS_NONE': {
            'type': 'object',
            'properties': {
                'type': {
                    'type': 'string',
                    'enum': ['IS_NONE'],
                },
                'source': {
                    '$ref': '#/check_definitions/EXPRESSION',
                },
            },
            'additionalProperties': False,
            'required': ['type', 'source'],
        },
        'SIZE': {
            'type': 'object',
            'properties': {
                'type': {
                    'type': 'string',
                    'enum': ['SIZE'],
                },
                'source': {
                    'oneOf': [
                        {'$ref': '#/check_definitions/UNION'},
                        {'$ref': '#/check_definitions/MULTIUNION'},
                    ],
                },
            },
            'additionalProperties': False,
            'required': ['type', 'source'],
        },
        'EQUAL': {
            'type': 'object',
            'properties': {
                'type': {
                    'type': 'string',
                    'enum': ['EQUAL'],
                },
                'sources': {
                    'type': 'array',
                    'items': {
                        '$ref': '#/check_definitions/OPERATORS',
                    },
                    'minItems': 2,
                    'maxItems': 2,
                },
            },
            'additionalProperties': False,
            'required': ['type', 'sources'],
        },
        'NOT': {
            'type': 'object',
            'properties': {
                'type': {
                    'type': 'string',
                    'enum': ['NOT'],
                },
                'source': {
                    '$ref': '#/check_definitions/EXPRESSION',
                },
            },
            'additionalProperties': False,
            'required': ['type', 'source'],
        },
        'OR': {
            'type': 'object',
            'properties': {
                'type': {
                    'type': 'string',
                    'enum': ['OR'],
                },
                'sources': {
                    'type': 'array',
                    'items': {
                        '$ref': '#/check_definitions/EXPRESSION',
                    },
                    'minItems': 1,
                },
            },
            'additionalProperties': False,
            'required': ['type', 'sources'],
        },
        'AND': {
            'type': 'object',
            'properties': {
                'type': {
                    'type': 'string',
                    'enum': ['AND'],
                },
                'sources': {
                    'type': 'array',
                    'items': {
                        '$ref': '#/check_definitions/EXPRESSION',
                    },
                    'minItems': 1,
                },
            },
            'additionalProperties': False,
            'required': ['type', 'sources'],
        },
        'DIFFERENT': {
            'type': 'object',
            'properties': {
                'type': {
                    'type': 'string',
                    'enum': ['DIFFERENT'],
                },
                'sources': {
                    'type': 'array',
                    'items': {
                        '$ref': '#/check_definitions/EXPRESSION',
                    },
                },
            },
            'additionalProperties': False,
            'required': ['type', 'sources'],
        },
        'MORE': {
            'type': 'object',
            'properties': {
                'type': {
                    'type': 'string',
                    'enum': ['MORE'],
                },
                'sources': {
                    'type': 'array',
                    'items': {
                        '$ref': '#/check_definitions/EXPRESSION',
                    },
                    'minItems': 2,
                    'maxItems': 2,
                },
            },
            'additionalProperties': False,
            'required': ['type', 'sources'],
        },
        'EQMORE': {
            'type': 'object',
            'properties': {
                'type': {
                    'type': 'string',
                    'enum': ['EQMORE'],
                },
                'sources': {
                    'type': 'array',
                    'items': {
                        '$ref': '#/check_definitions/EXPRESSION',
                    },
                    'minItems': 2,
                    'maxItems': 2,
                },
            },
            'additionalProperties': False,
            'required': ['type', 'sources'],
        },
        'LESS': {
            'type': 'object',
            'properties': {
                'type': {
                    'type': 'string',
                    'enum': ['LESS'],
                },
                'sources': {
                    'type': 'array',
                    'items': {
                        '$ref': '#/check_definitions/EXPRESSION',
                    },
                    'minItems': 2,
                    'maxItems': 2,
                },
            },
            'additionalProperties': False,
            'required': ['type', 'sources'],
        },
        'EQLESS': {
            'type': 'object',
            'properties': {
                'type': {
                    'type': 'string',
                    'enum': ['EQLESS'],
                },
                'sources': {
                    'type': 'array',
                    'items': {
                        '$ref': '#/check_definitions/EXPRESSION',
                    },
                    'minItems': 2,
                    'maxItems': 2,
                },
            },
            'additionalProperties': False,
            'required': ['type', 'sources'],
        },
        'IS_SUBSET': {
            'type': 'object',
            'properties': {
                'type': {
                    'type': 'string',
                    'enum': ['IS_SUBSET'],
                },
                'sources': {
                    'type': 'array',
                    'items': {
                        'oneOf': [
                            {'$ref': '#/check_definitions/UNION'},
                            {'$ref': '#/check_definitions/MULTIUNION'},
                        ],
                    },
                    'minItems': 2,
                    'maxItems': 2,
                },
            },
            'additionalProperties': False,
            'required': ['type', 'sources'],
        },
        'UNIQUE_ITEMS': {
            'type': 'object',
            'properties': {
                'type': {
                    'type': 'string',
                    'enum': ['UNIQUE_ITEMS'],
                },
                'sources': {
                    'type': 'array',
                    'items': {
                        'oneOf': [
                            {'$ref': '#/check_definitions/UNION'},
                            {'$ref': '#/check_definitions/MULTIUNION'},
                        ],
                    },
                    'minItems': 2,
                    'maxItems': 2,
                },
            },
            'additionalProperties': False,
            'required': ['type', 'sources'],
        },
        'NONE': {
            'type': 'object',
            'properties': {
                'type': {
                    'type': 'string',
                    'enum': ['NONE'],
                },
            },
            'additionalProperties': False,
            'required': ['type'],
        },
        'NUMBER': {
            'type': 'object',
            'properties': {
                'type': {
                    'type': 'string',
                    'enum': ['NUMBER'],
                },
                'source': {
                    'type': 'number',
                },
            },
            'additionalProperties': False,
            'required': ['type', 'source'],
        },
        'RATIONAL': {
            'type': 'object',
            'properties': {
                'type': {
                    'type': 'string',
                    'enum': ['RATIONAL'],
                },
                'source': {
                    'type': 'string',
                },
            },
            'additionalProperties': False,
            'required': ['type', 'source'],
        },
        'STRING': {
            'type': 'object',
            'properties': {
                'type': {
                    'type': 'string',
                    'enum': ['STRING'],
                },
                'source': {
                    'type': 'string',
                },
            },
            'additionalProperties': False,
            'required': ['type', 'source'],
        },
        'BOOLEAN': {
            'type': 'object',
            'properties': {
                'type': {
                    'type': 'string',
                    'enum': ['BOOLEAN'],
                },
                'source': {
                    'type': 'boolean',
                },
            },
            'additionalProperties': False,
            'required': ['type', 'source'],
        },
        'SUM': {
            'type': 'object',
            'properties': {
                'type': {
                    'type': 'string',
                    'enum': ['SUM'],
                },
                'sources': {
                    'type': 'array',
                    'items': {
                        '$ref': '#/check_definitions/EXPRESSION',
                    },
                    'minItems': 2,
                },
            },
            'additionalProperties': False,
            'required': ['type', 'sources'],
        },
        'MINUS': {
            'type': 'object',
            'properties': {
                'type': {
                    'type': 'string',
                    'enum': ['MINUS'],
                },
                'sources': {
                    'type': 'array',
                    'items': {
                        '$ref': '#/check_definitions/EXPRESSION',
                    },
                    'minItems': 2,
                    'maxItems': 2,
                },
            },
            'additionalProperties': False,
            'required': ['type', 'sources'],
        },
        'MULT': {
            'type': 'object',
            'properties': {
                'type': {
                    'type': 'string',
                    'enum': ['MULT'],
                },
                'sources': {
                    'type': 'array',
                    'items': {
                        '$ref': '#/check_definitions/EXPRESSION',
                    },
                    'minItems': 2,
                },
            },
            'additionalProperties': False,
            'required': ['type', 'sources'],
        },
        'REMAINDER': {
            'type': 'object',
            'properties': {
                'type': {
                    'type': 'string',
                    'enum': ['REMAINDER'],
                },
                'sources': {
                    'type': 'array',
                    'items': {
                        '$ref': '#/check_definitions/EXPRESSION',
                    },
                    'minItems': 2,
                    'maxItems': 2,
                },
            },
            'additionalProperties': False,
            'required': ['type', 'sources'],
        },
        'QUOTIENT': {
            'type': 'object',
            'properties': {
                'type': {
                    'type': 'string',
                    'enum': ['QUOTIENT'],
                },
                'sources': {
                    'type': 'array',
                    'items': {
                        '$ref': '#/check_definitions/EXPRESSION',
                    },
                    'minItems': 2,
                    'maxItems': 2,
                },
            },
            'additionalProperties': False,
            'required': ['type', 'sources'],
        },
        'UNION': {
            'type': 'object',
            'properties': {
                'type': {
                    'type': 'string',
                    'enum': ['UNION'],
                },
                'sources': {
                    'type': 'array',
                    'items': {
                        'oneOf': [
                            {'$ref': '#/check_definitions/EXPRESSION'},
                            {'$ref': '#/check_definitions/UNION'},
                        ],
                    },
                    'minItems': 1,
                },
            },
            'additionalProperties': False,
            'required': ['type', 'sources'],
        },
        'MULTIUNION': {
            'type': 'object',
            'properties': {
                'type': {
                    'type': 'string',
                    'enum': ['MULTIUNION'],
                },
                'sources': {
                    'type': 'array',
                    'items': {
                        'oneOf': [
                            {'$ref': '#/check_definitions/EXPRESSION'},
                            {'$ref': '#/check_definitions/UNION'},
                        ],
                    },
                    'minItems': 1,
                },
            },
            'additionalProperties': False,
            'required': ['type', 'sources'],
        },
        'IS_PERMUTATION_FROM': {
            'type': 'object',
            'properties': {
                'type': {
                    'type': 'string',
                    'enum': ['IS_PERMUTATION_FROM'],
                },
                'sources': {
                    'type': 'array',
                    'items': {
                        '$ref': '#/check_definitions/MULTIUNION'
                    },
                    'minItems': 2,
                    'maxItems': 2,
                }
            }
        },
    }

    # нужно использовать в json-схеме с ключом `check_definitions`
    @classmethod
    def definitions(cls):
        return deepcopy(cls.COMMON_DEFINITIONS)

    def check_tree(self, expression_tree):
        """
        Шаг прохождения по дереву выражения при проверке ответа

        :param expression_tree: текущее дерево выражения
        :return: правильность ответа
        :rtype: bool
        """
        if not isinstance(expression_tree, dict):
            return expression_tree
        if 'sources' in expression_tree:
            return getattr(self, 't_{0}'.format(expression_tree['type']))(*[
                self.check_tree(source)
                for source in expression_tree['sources']
            ])
        elif 'source' in expression_tree:
            return getattr(self, 't_{0}'.format(expression_tree['type']))(
                self.check_tree(expression_tree['source']))
        else:
            return getattr(self, 't_{0}'.format(expression_tree['type']))()

    # type operators

    def t_NONE(self):
        return None

    def t_BOOLEAN(self, arg):
        return bool(arg)

    def t_NUMBER(self, arg):
        return float(arg)

    def t_RATIONAL(self, arg):
        return InlineChecker.rational_conversion(arg)

    def t_STRING(self, arg):
        return str(arg)

    def t_UNION(self, *args):
        """Возвращает множество из аргументов"""
        return frozenset(args)

    def t_MULTIUNION(self, *args):
        """
        Возвращает мультимножество в виде `collections.Counter` из аргументов
        """
        return Counter(args)

    def t_IS_PERMUTATION_FROM(self, *args):
        """
        первый параметр - мультимножество
        второй параметр - мультимножество множеств
        """
        answer, valid_answers = args

        if sum(answer.values()) > 50 or sum(valid_answers.values()) > 50:
            return False

        graph = generate_graph_from_counters(*args)
        if not graph:
            return False

        vertex_count = len(graph)
        matching_size, _ = kuhn(graph, vertex_count, vertex_count)
        return matching_size == vertex_count

    def t_SIZE(self, arg):
        """
        Возвращает количество элементов, отличных от NONE,
        в множестве или мультимноестве.
        """
        collection = arg
        if isinstance(arg, Counter):
            collection = arg.elements()

        return len([x for x in collection if x is not None])

    # unary operators

    def t_IS_NONE(self, arg):
        return arg is None

    def t_NOT(self, arg):
        return not arg

    # multi arg operators

    def t_OR(self, *args):
        return reduce(operator.or_, args)

    def t_AND(self, *args):
        return reduce(operator.and_, args)

    def t_MORE(self, x, y):
        return x > y

    def t_EQMORE(self, x, y):
        return x >= y

    def t_LESS(self, x, y):
        return x < y

    def t_EQLESS(self, x, y):
        return x <= y

    def t_EQUAL(self, x, y):
        if type(x) is int:
            x = float(x)
        if type(y) is int:
            y = float(y)
        if type(x) is not type(y):
            return False
        if type(x) is float and type(y) is float:
            # Возвращаем результат сравнения х и у с точночтью 1e-9
            return abs(x - y) <= max(1e-9 * max(abs(x), abs(y)), 0.0)
        return x == y

    def t_IS_SUBSET(self, x, y):
        """
        Проверка, что первое (мульти-)множество является подмножеством второго
        """
        if type(x) is not type(y):
            return False
        elif isinstance(x, set):
            return x.issubset(y)
        else:
            return not (x - y)

    def t_UNIQUE_ITEMS(self, x, y):
        """
        Проверка, что каждое (мульти-)множество содержит элемент, не входящий
        в другое множество
        """
        if type(x) is type(y):
            return bool((x - y) and (y - x))
        else:
            return False

    def t_SUM(self, *args):
        """
        `None` считается равным 0
        """
        return reduce(operator.add, [_f for _f in args if _f], 0)

    def t_MINUS(self, x, y):
        """
        Возвращает разность двух чисел
        """
        return x - y

    def t_MULT(self, *args):
        """
        `None` считается равням 0
        """
        return reduce(operator.mul, [0 if x is None else x for x in args])

    def t_DIFFERENT(self, *args):
        return len(set(args)) == len(args)

    def t_REMAINDER(self, x, y):
        """
        Возвращает остаток от деления целых чисел по модулю
        """
        for arg in [x, y]:
            if arg is None or int(arg) != arg:
                return None

        if y == 0:
            return None

        result = x % y

        # python не совсем правильно считает остаток
        # по определению остаток не может быть отрицательным
        # но python, в случае отрицательного второго аргумента
        # возвращает отрицательный остаток
        if result < 0:
            result -= y

        return result

    def t_QUOTIENT(self, x, y):
        """
        Возвращает целую часть от деления числа по модулю
        """
        for arg in [x, y]:
            if arg is None or int(arg) != arg:
                return None

        if y == 0:
            return None

        # Сложности с делением отрицательных чисел в python
        # Сначала правильно вычисляем остаток от деления (из расчета, что он
        # не может быть меньше нуля), после этого делим уже без остатка
        r = self.t_REMAINDER(x, y)
        return old_div((x - r), y)


class DragChecker(CommonChecker):
    """
    Класс для проверки ответов маркеров `dragimage` и `dragtable`
    """
    @classmethod
    def definitions(cls):
        """Добавляет определение оператора `FIELD`"""
        drag_definitions = CommonChecker.definitions()
        drag_definitions['FIELD'] = {
            'type': 'object',
            'properties': {
                'type': {
                    'type': 'string',
                    'enum': ['FIELD'],
                },
                'source': {
                    'type': 'integer',
                },
            },
            'additionalProperties': False,
            'required': ['type', 'source'],
        }
        drag_definitions['EXPRESSION']['oneOf'].append(
            {'$ref': '#/check_definitions/FIELD'})
        drag_definitions['OPERATORS']['oneOf'].append(
            {'$ref': '#/check_definitions/FIELD'})

        return drag_definitions

    def __init__(self, choices, answer):
        """
        Инициализация класса проверки

        :param choices: поле `choices` маркеров перетаскивания
        :param answer: ответ ученика
        """
        self.choices = {
            choice['id']: choice['value']
            for choice in choices
        }
        self.answer = answer

    def t_FIELD(self, arg):
        try:
            return self.choices[self.answer[str(arg)]]
        except KeyError:
            return None


class InlineChecker(CommonChecker):
    """Класс для проверки маркера `inline`"""

    @classmethod
    def definitions(cls):
        """Добавляет определение оператора `INPUT`"""
        inline_definitions = CommonChecker.definitions()
        inline_definitions['INPUT'] = {
            'type': 'object',
            'properties': {
                'type': {
                    'type': 'string',
                    'enum': ['INPUT'],
                },
                'source': {
                    'type': 'integer',
                },
            },
            'additionalProperties': False,
            'required': ['type', 'source'],
        }
        inline_definitions['EXPRESSION']['oneOf'].append(
            {'$ref': '#/check_definitions/INPUT'})
        inline_definitions['OPERATORS']['oneOf'].append(
            {'$ref': '#/check_definitions/INPUT'})

        return inline_definitions

    class MissingAnswerException(Exception):
        """
        Сигнализирует о том, что ответа на инпут не было - нужно выставить
        False проверке всей группы
        """
        pass

    class InvalidNumberException(Exception):
        """
        Сигнализирует о том, что не получилось привести к числу ввод в инпут
        типа 'field' либо один из аргументов оператора `EQUAL_NUMBERS` - нужно
        выставить False проверке всей группы
        """
        pass

    class InvalidRationalException(Exception):
        """
        Сигнализирует о том, что не получилось привести к дробному числу
        (Fraction) ввод в инпут типа 'rational'
        """
        pass

    def __init__(self, inputs, expressions, answer):
        """
        Инициализация класса проверки

        :param inputs: поле `inputs` маркера - словарб полей для ввода
        :param expressions: поле `check` маркера - маппинг id_группы: дерево
        :param answer: ответ ученика
        """
        self.inputs = inputs
        self.expressions = expressions
        self.answer = answer

    def check(self):
        """
        Верхний этап проверки - запуск рекурсивной проверки по каждому дереву
        из `expressions`.
        Обработка ошибок `MissingAnswerException`, `InvalidNumberException` и
        `InvalidRationalException` - выставляет `False` в проверке дерева,
        где было получено исключение.
        """
        checked = {}
        for group_id, group_check_tree in self.expressions.items():
            try:
                checked[group_id] = self.check_tree(group_check_tree)
            except (
                self.MissingAnswerException,
                self.InvalidNumberException,
                self.InvalidRationalException,
            ):
                checked[group_id] = False
        return checked

    def field_conversion(self, _input, answer):
        """Дополнительное приведение для 'field'"""
        answer = str(answer).strip()
        if _input['options']['type_content'] == 'number':
            answer = answer.replace(',', '.').replace(' ', '')
            # преобразуем ответ вида (-2) к -2
            # при этом, случаи типа -(-2) или +(-2) не трогаем
            # тесты: test_convert_answers: field_with_parenthesis
            if answer and answer[0] == '(' and answer[-1] == ')':
                answer = answer[1:-1]
            try:
                answer = float(answer)
            except (ValueError, TypeError):
                raise self.InvalidNumberException
            return answer

        elif _input['options']['type_content'] == 'text':
            return answer.lower().replace(u'ё', u'е')
        elif _input['options']['type_content'] == 'spaceless':
            return re.sub('\s', '', answer)
        return answer

    @staticmethod
    def rational_conversion(answer):
        """Дополнительное приведение для 'rational'"""
        if RE_DECIMAL_FRACTION.match(answer):
            # Ответ вида '123' или '-1.23'
            answer = answer.replace(',', '.').replace(' ', '')
            try:
                return Fraction(answer)
            except Exception:
                raise InlineChecker.InvalidRationalException
        elif RE_MIXED_FRACTION.match(answer):
            # Ответ вида '-1 1/2' или '1/2'
            answer = answer.replace('/', ' ').split()
            integer_part = 0

            # Проверяем, есть ли целая часть в ответе
            if len(answer) == 3:
                integer_part = Fraction(answer[0])
                answer.pop(0)

            fraction_part = Fraction('/'.join(answer))

            # Суммируем дробную и целую часть, учитывая знак
            if integer_part < 0:
                # Пример -1 1/2 = -(1 + 1/2). Если не вынести минус за скобки,
                # то получим неправильный ответ. -1 + 1/2 = -1/2
                return -(-integer_part + fraction_part)
            else:
                return integer_part + fraction_part
        else:
            raise InlineChecker.InvalidRationalException

    def t_INPUT(self, input_id):
        """
        Получает ввод из поля. Если его нет или он `None` - поднимает ошибку.
        Для ввода в 'field' и 'rational' дополнительно делаем нормализацию
        ответа и приведение к нужному типу
        """
        input_id = str(input_id)
        if input_id not in self.answer:
            raise self.MissingAnswerException
        answer = self.answer[input_id]
        if answer is None:
            return

        # Дополнительное приведение для типов инпута 'field' и 'rational'
        _input = self.inputs[input_id]
        if _input['type'] == 'field':
            answer = self.field_conversion(_input, answer)
        elif _input['type'] == 'rational':
            answer = self.rational_conversion(answer)
        return answer
