from builtins import object

import jsonschema

from django.core.exceptions import ValidationError as DjangoValidationError

from kelvin.problems.constants import ANDROID_ONLY_MARKER_TYPES


class BaseMarker(object):
    """
    Интерфейс маркеров
    """
    CORRECT = 1
    INCORRECT = 0
    SKIPPED = -1
    UNCHECKED = 2

    SKIPPED_ANSWER = None

    TYPE_NAME = None
    JSON_SCHEMA_OPTIONS = {}
    JSON_SCHEMA_EXTRA = {}

    ANSWER_JSON_SCHEMA = {}  # схема ответа пользователя
    RIGHT_ANSWER_JSON_SCHEMA = {}  # схема правильного ответа для маркера
    CHECKS_JSON_SCHEMA = {}

    checkable = True

    def __init__(self, data, answer, checks=None):
        """
        :param data: данные маркера
        :type data: dict
        """
        self.data = data
        self.answer = answer
        self.checks = checks

    @property
    def is_only_android(self):
        """
        Маркер является только андроидным и не проверяется у нас
        """
        return self.data['type'] in ANDROID_ONLY_MARKER_TYPES

    @property
    def max_mistakes(self):
        """
        Максимальное число ошибок в маркере
        """
        raise NotImplementedError('Should be written in subclass')

    @property
    def json_schema(self):
        """
        Возвращает json схему маркера
        """
        if not self.TYPE_NAME:
            return {}
        schema = {
            'type': 'object',
            'properties': {
                'id': {'type': 'number'},
                'type': {
                    'enum': [self.TYPE_NAME],
                },
                'options': self.JSON_SCHEMA_OPTIONS,
            },
            'additionalProperties': False,
            'required': ['type', 'options'],
        }
        schema.update(self.JSON_SCHEMA_EXTRA)
        return schema

    def check(self, user_answer):
        """
        Проверка ответа пользователя. Должна быть переопредела у детей

        :param user_answer:  ответ пользователя
        :type user_answer: Answer
        :return:  (статус правильности, количество ошибок)
        :rtype: tuple (int, int)
        """
        raise NotImplementedError('Should be written in subclass')

    def get_embedded_objects(self):
        """
        Вернуть все формулы и ресурсы, указанные в маркере, в списке
        возможны дубликаты

        :return: список кортежей (тип объекта, идентификатор объекта)
        :rtype: list [tuple (str, str)]
        """
        return []

    def is_skipped(self, user_answer):
        """
        Проверка, является ли ответ на вопрос пропуском маркера
        """
        return user_answer == self.SKIPPED_ANSWER

    def validate(self):
        """
        Проверяет правильность данных
        """
        try:
            jsonschema.validate(self.data, self.json_schema)
            if self.answer is not None:
                jsonschema.validate(self.answer, self.RIGHT_ANSWER_JSON_SCHEMA)
            if self.checks:
                jsonschema.validate(self.checks, self.CHECKS_JSON_SCHEMA)
        except jsonschema.ValidationError as e:
            raise DjangoValidationError(e.message)

    def validate_answer(self, answer):
        """
        Проверяет правильность формата ответа
        """
        if answer is None:
            return

        try:
            jsonschema.validate(answer, self.ANSWER_JSON_SCHEMA)
        except jsonschema.ValidationError as e:
            raise DjangoValidationError(e.message)

    # TODO может появиться `normalize_answer`
