from django.core.exceptions import ValidationError as DjangoValidationError

from kelvin.problems.constants import RESOURCE_FORMULA_RE
from kelvin.problems.markers import BaseMarker


class ChoiceMarker(BaseMarker):
    """
    Маркер выбора
    """
    TYPE_NAME = 'choice'
    FLAVOR_CHECKBOX = 'checkbox'
    FLAVOR_RADIO = 'radio'

    ANSWER_JSON_SCHEMA = {
        'oneOf': [
            {'enum': [BaseMarker.SKIPPED_ANSWER]},
            {'type': 'array'},
        ],
    }
    RIGHT_ANSWER_JSON_SCHEMA = ANSWER_JSON_SCHEMA
    JSON_SCHEMA_OPTIONS = {
        'type': 'object',
        'properties': {
            'choices': {
                'type': 'array',
                'items': {
                    'type': 'string',
                },
            },
            'flavor': {
                'enum': [FLAVOR_CHECKBOX, FLAVOR_RADIO],
            },
            'type_display': {
                'enum': [
                    None,
                    'table',
                    'vertical',
                    'horizontal',
                    'absolute',
                    '2cols',
                    '3cols',
                ],
            },
        },
        'additionalProperties': False,
        'required': ['choices'],
    }

    def __init__(self, *args, **kwargs):
        """
        Заполняем `self.flavor` - одиночный (`FLAVOR_RADIO`)
        или множественный (`FLAVOR_CHECKBOX`) выбор.
        По умолчанию `FLAVOR_CHECKBOX`.
        """
        super(ChoiceMarker, self).__init__(*args, **kwargs)

        self.flavor = self.data['options'].get('flavor', self.FLAVOR_CHECKBOX)

    def check(self, user_answer):
        """
        Для множественного выбора число ошибок считаем, как сумму числа
        неуказанных правильных ответов и числа указанных неправильных ответов.
        Для одиночного выбора 0 ошибок при правильном ответе и одна ошибка
        при неправильном.
        Для статуса: возвращает список, в котором для каждого из индексов
        ответа пользователя проставлена корректность
        Пример:
        для user_answer = [1,3,5]
            self.data['answer'] = [3,5]
        вернёт [0,1,1]
        :param user_answer: ответ пользователя, список отмеченных вариантов
        :return: статус ответа, количество ошибок
        """
        if self.is_skipped(user_answer):
            return self.SKIPPED, len(self.answer)
        answer_status = []
        mistakes = 0
        right_answers = set(self.answer)
        for answer in user_answer:
            if answer in right_answers:
                answer_status.append(self.CORRECT)
            else:
                answer_status.append(self.INCORRECT)
                mistakes += 1

        if self.flavor == self.FLAVOR_CHECKBOX:
            # Добавляет ошибок за правильные ответы, которые не указаны
            mistakes += len(right_answers) - len(user_answer) + mistakes
        return answer_status, mistakes

    def get_embedded_objects(self):
        """
        Объекты могут быть внутри `options.choices`
        """
        objects = []
        choices = self.data['options'].get('choices', [])
        for choice in choices:
            objects.extend(RESOURCE_FORMULA_RE.findall(choice))
        return objects

    @property
    def max_mistakes(self):
        """
        Для одиночного выбора максимум одна ошибка.
        Для множественного - количество вариантов, но не меньше 1
        """
        if self.flavor == self.FLAVOR_RADIO:
            return 1
        return len(self.data['options']['choices']) or 1

    def validate(self):
        """
        Добавляем проверку ответа в случае одиночного выбора
        """
        super(ChoiceMarker, self).validate()

        if self.flavor == self.FLAVOR_RADIO:
            self.validate_radio_answer(self.answer)

    def validate_answer(self, answer):
        """
        Добавляем проверку ответа пользователя в случае одиночного выбора
        """
        super(ChoiceMarker, self).validate_answer(answer)

        if self.flavor == self.FLAVOR_RADIO:
            self.validate_radio_answer(answer)

    def validate_radio_answer(self, answer):
        """
        Проверяет, что ответ на одиночный выбор содержит только одно значение
        """
        if not self.is_skipped(answer) and len(answer) != 1:
            raise DjangoValidationError('Choice marker with `{0}` flavor '
                                        'can only have one right answer'
                                        .format(self.FLAVOR_RADIO))
