from django.core.exceptions import ValidationError as DjangoValidationError

from kelvin.problems.markers import BaseMarker


class ChooseImageMarker(BaseMarker):
    """
    Маркер выбора областей
    """
    TYPE_NAME = 'chooseimage'
    ANSWER_JSON_SCHEMA = {
        'oneOf': [
            {'enum': [BaseMarker.SKIPPED_ANSWER]},
            {
                'type': 'array',
                'items': {
                    'type': 'string',
                },
            },
        ],
    }
    RIGHT_ANSWER_JSON_SCHEMA = {
        'type': 'array',
        'items': ANSWER_JSON_SCHEMA,
    }
    JSON_SCHEMA_OPTIONS = {
        'type': 'object',
        'properties': {
            'hit_areas': {
                'type': 'array',
                'items': {
                    'type': 'object',
                    'properties': {
                        'x': {
                            'type': 'number',
                        },
                        'y': {
                            'type': 'number',
                        },
                        'width': {
                            'type': 'number',
                        },
                        'height': {
                            'type': 'number',
                        },
                        'id': {
                            'type': 'string',
                        },
                    },
                },
            },
            'image_file': {
                'type': 'integer',
            },
        },
        'additionalProperties': False,
        'required': [
            'hit_areas',
            'image_file',
        ],
    }

    @property
    def max_mistakes(self):
        """
        Пока никак не считаем число ошибок, только правильно или неправильно
        """
        return 1

    def _compare_answers(self, answer, right_answer):
        """
        Сравнение ответа пользователя с правильным ответом
        """
        mistakes = 0
        status = []
        right_answer = set(right_answer)
        for item in answer:
            if item in right_answer:
                status.append(self.CORRECT)
                right_answer.remove(item)
            else:
                status.append(self.INCORRECT)
                mistakes += 1
        # к явным ошибкам добавляем правильные ответы, которые не были даны
        return status, mistakes + len(right_answer)

    def check(self, user_answer):
        """
        Проверяет на совпадение с одним из правильных вариантов ответа.

        В статусе пишет с каким вариантом ответа сравнивали и правильность
        ответа пользователя. При неправильном ответе возвращает вариант
        сравнения с наименьшими "ошибками".
        """
        if self.is_skipped(user_answer):
            return (
                # считаем, что сравнивали с первым ответом
                {'compared_with': 0, 'status': self.SKIPPED},
                1,
            )
        answer_statuses = []
        for compared_with, right_answer in enumerate(self.answer):
            answer_status, mistakes = self._compare_answers(
                user_answer,
                right_answer,
            )
            if not mistakes:
                return (
                    {
                        'compared_with': compared_with,
                        'status': answer_status,
                    },
                    0,
                )
            answer_statuses.append((answer_status, mistakes, compared_with))

        if not answer_statuses:
            raise DjangoValidationError("Strange case. Answer statuses are empty - possible heat_areas misconfig.")

        status = min(answer_statuses, key=lambda tuple_: tuple_[1])
        return (
            {
                'compared_with': status[2],
                'status': status[0],
            },
            1,
        )

    def get_embedded_objects(self):
        """
        Ресурс указан в поле `options.image_file`
        """
        return [
            ('resource', self.data['options']['image_file']),
        ]

    def validate(self):
        used_ids = set([])
        for hit_area in self.data.get("hit_areas", []):
            if hit_area["id"] in used_ids:
                raise DjangoValidationError("hit_area id {} is not unique in this problem".format(hit_area["id"]))
        super(ChooseImageMarker, self).validate()
