from kelvin.problems.checkers import DragChecker
from kelvin.problems.markers import BaseMarker


class DragImagesMarker(BaseMarker):
    """
    Маркер перетаскивания на картинке
    """
    TYPE_NAME = 'dragimage'
    ANSWER_JSON_SCHEMA = {
        'type': 'object',
        'patternProperties': {
            '^\d+$': {
                'type': 'integer',
            },
        },
        'additionalProperties': False,
    }
    RIGHT_ANSWER_JSON_SCHEMA = ANSWER_JSON_SCHEMA
    CHECKS_JSON_SCHEMA = {
        'check_definitions': DragChecker.definitions(),
        '$ref': '#/check_definitions/BOOL_VALUE_OPERATORS',
    }
    JSON_SCHEMA_OPTIONS = {
        'type': 'object',
        'properties': {
            'choices': {
                'type': 'array',
                'items': {
                    'type': 'object',
                    'properties': {
                        'id': {
                            'type': 'integer',
                        },
                        'value': {
                            'oneOf': [
                                {'type': 'integer'},
                                {'type': 'boolean'},
                            ],
                        },
                        'resource_id': {
                            'type': 'integer',
                        },
                        'x': {
                            'type': 'number',
                        },
                        'y': {
                            'type': 'number',
                        },
                        'width': {
                            'type': 'number',
                        },
                        'height': {
                            'type': 'number',
                        },
                    },
                    'additionalProperties': False,
                    'required': ['id', 'value', 'resource_id'],
                },
            },
            'multiple_choices': {
                'type': 'boolean',
            },
            'gravity_radius': {
                'type': 'number',
            },
            'main_image': {
                '$ref': '#/definitions/image_object',
            },
            'fields': {
                'type': 'array',
                'items': {
                    '$ref': '#/definitions/field_object',
                },
            },
        },
        'additionalProperties': False,
        'required': ['choices', 'multiple_choices',
                     'main_image', 'fields', 'gravity_radius'],
    }
    JSON_SCHEMA_EXTRA = {
        'definitions': {
            'image_object': {
                'type': 'object',
                'properties': {
                    'resource_id': {
                        'type': 'integer',
                    },
                    'x': {
                        'type': 'number',
                    },
                    'y': {
                        'type': 'number',
                    },
                    'width': {
                        'type': 'number',
                    },
                    'height': {
                        'type': 'number',
                    },
                },
                'additionalProperties': False,
                'required': ['resource_id', 'x', 'y', 'width'],
            },
            'field_object': {
                'type': 'object',
                'properties': {
                    'id': {
                        'type': 'integer',
                    },
                    'x': {
                        'type': 'number',
                    },
                    'y': {
                        'type': 'number',
                    },
                    'disabled': {
                        'type': 'boolean',
                    },
                    'choice': {
                        'type': 'integer',
                    },
                    'gravity': {
                        'type': 'string',
                    },
                },
                'additionalProperties': False,
                'required': ['id', 'x', 'y'],
            },
        }
    }

    def check(self, user_answer):
        """
        Если правильно, то нет ошибок, иначе 1 ошибка
        """
        if self.is_skipped(user_answer):
            return self.SKIPPED, 1
        is_right = DragChecker(
            self.data['options']['choices'],
            user_answer,
        ).check_tree(self.checks)
        if is_right:
            return self.CORRECT, 0
        return self.INCORRECT, 1

    @property
    def max_mistakes(self):
        """
        Максимум 1 ошибка
        """
        return 1

    def get_embedded_objects(self):
        """
        Ресурсы маркера
        """
        resources = [self.data['options']['main_image']['resource_id']]
        for item in self.data['options']['choices']:
            resources.append(item['resource_id'])
        return [('resource', resource_id) for resource_id in resources]
