from django.core.exceptions import ValidationError as DjangoValidationError

from kelvin.problems.markers import BaseMarker

HIGHLIGHT_TYPES = {
    'highlight': [
        'highlight/word',
        'highlight/letter',
        'highlight/stress',
    ],
    'morpheme': [
        'morpheme/root',
        'morpheme/suffix',
        'morpheme/prefix',
        'morpheme/stem',
        'morpheme/ending',
    ],
    'morpheme-with-bubble': [
        'morpheme-with-bubble/root',
        'morpheme-with-bubble/suffix',
        'morpheme-with-bubble/prefix',
        'morpheme-with-bubble/stem',
        'morpheme-with-bubble/ending',
    ],
}
HIGHLIGHT_TYPES_SCHEMA = {
    'type': 'object',
    'properties': {
        key: {
            "type": "array",
            "items": {
                "type": "string",
                "enum": HIGHLIGHT_TYPES[key],
            },
        } for key in HIGHLIGHT_TYPES
    },
    'additionalProperties': False,
}
HIGHLIGHT_SCHEMA = {
    'type': 'array',
    'items': {
        'type': 'array',
        'items': {
            'type': 'integer',
            'minimum': 0,
        },
        'minItems': 2,
        'maxItems': 2,
    },
}
ANSWER_PROPERTIES_SCHEMA = {
    highlight: HIGHLIGHT_SCHEMA
    for key in HIGHLIGHT_TYPES
    for highlight in HIGHLIGHT_TYPES[key]
}


def compare_ranges(range1, range2):
    return range1[0] == range2[0] and range1[1] == range2[1]


def has_range(pos_range, collection):
    return bool(
        [another_range for another_range in collection if compare_ranges(pos_range, another_range)]
    )


class HighlightMarker(BaseMarker):
    """
    Маркер выделений
    """
    TYPE_NAME = 'highlight'
    ANSWER_JSON_SCHEMA = {
        'type': 'object',
        'properties': ANSWER_PROPERTIES_SCHEMA,
        'additionalProperties': False,
    }

    RIGHT_ANSWER_JSON_SCHEMA = {
        'type': 'array',
        'items': ANSWER_JSON_SCHEMA,
        'minItems': 1,
    }
    JSON_SCHEMA_OPTIONS = {
        'type': 'object',
        'properties': {
            'text': {
                'type': 'string',
            },
            'highlight_types': HIGHLIGHT_TYPES_SCHEMA,
        },
        'additionalProperties': False,
        'required': ['text', 'highlight_types'],
    }

    def validate(self):
        super(HighlightMarker, self).validate()

        self._validate_answer_highlights()
        self._validate_answer_ranges()

    def _validate_answer_highlights(self):
        """
        Проверка на то, что в ответе не существует
        типов выделений отличных от указаных в разметке проблемы
        """
        data_highlight_types = self.data['options']['highlight_types']
        data_highlights = [
            highlight
            for key in data_highlight_types
            for highlight in data_highlight_types[key]
        ]
        for answer in self.answer:
            answer_highlights = [highlight for highlight in answer]
            if set(answer_highlights) - set(data_highlights):
                raise DjangoValidationError(
                    'Answer types don\'t match problem\'s markup'
                )

    def _validate_answer_ranges(self):
        """
        Проверка на то, что в ответе все полуинтервалы
        выделений - верные
        """
        for answer in self.answer:
            for answer_ranges in list(answer.values()):
                for pos_range in answer_ranges:
                    if pos_range[0] > pos_range[1]:
                        raise DjangoValidationError(
                            'range {0} is invalid'.format(pos_range)
                        )

    def check(self, user_answer):
        """
        https://docs.pelican.common.yandex.ru/formats/markers/highlight.html
        см. формат результата проверки пользовательского ответа

        :param user_answer: ответ пользователя (массив разборов слов)
        :return: статус ответа, количество ошибок
        """

        if self.is_skipped(user_answer):
            return self.SKIPPED, self.max_mistakes

        error_count = 0

        answer, check_results = self._get_max_fitting_answer(user_answer)

        for section, result_data in list(check_results.items()):
            incorrect_ranges = check_results[section].get('incorrect', [])
            skipped_ranges = check_results[section].get('skipped', [])
            error_count += len(incorrect_ranges) + len(skipped_ranges)

        check_status = self.INCORRECT if error_count else self.CORRECT

        status = {
            'status': check_status,
            'results': check_results,
        }

        return status, error_count

    def _get_max_fitting_answer(self, user_answer):
        max_fitting_count = 0
        max_fitting_answer = None
        max_fitting_check_results = {}

        for answer in self.answer:
            fitting_count = 0
            check_results = {}
            for section, ranges in list(answer.items()):
                if section not in user_answer:
                    check_results[section] = {
                        'skipped': ranges
                    }
                    continue

                incorrect_ranges = user_answer[section][:]

                if section not in check_results:
                    check_results[section] = {}

                for pos_range in ranges:
                    range_status = 'skipped'
                    if has_range(pos_range, user_answer[section]):
                        fitting_count += 1
                        incorrect_ranges = [
                            user_range for user_range in incorrect_ranges if not compare_ranges(user_range, pos_range)]
                        range_status = 'correct'

                    check_results[section][range_status] = (
                        check_results[section].get(range_status, [])
                    )
                    check_results[section][range_status].append(pos_range)

                if incorrect_ranges:
                    check_results[section]['incorrect'] = incorrect_ranges

            if max_fitting_answer is None or fitting_count > max_fitting_count:
                max_fitting_count = fitting_count
                max_fitting_check_results = check_results
                max_fitting_answer = answer

        return max_fitting_answer, max_fitting_check_results

    @property
    def max_mistakes(self):
        """
        Максимальное число ошибок - возвращаем всегда 1
        """
        return 1
