from kelvin.problems.checkers import InlineChecker
from kelvin.problems.constants import RESOURCE_FORMULA_RE
from kelvin.problems.markers import BaseMarker


class InlineMarker(BaseMarker):
    """
    Маркер связанного ввода
    """
    TYPE_NAME = 'inline'
    ANSWER_JSON_SCHEMA = {
        'type': 'object',
        'patternProperties': {
            '^\d+$': {
                'oneOf': [
                    {'type': 'null'},
                    {'type': 'integer'},
                    {'type': 'boolean'},
                    {'type': 'string'},
                ],
            },
        },
        'additionalProperties': False,
    }

    RIGHT_ANSWER_JSON_SCHEMA = {
        'type': 'object',
        'patternProperties': {
            '^\d+$': {
                'oneOf': [
                    {'type': 'null'},
                    {'type': 'integer'},
                    {'type': 'boolean'},
                    {'type': 'string'},
                    {
                        'type': 'array',
                        'items': {
                            'oneOf': [
                                {'type': 'null'},
                                {'type': 'integer'},
                                {'type': 'boolean'},
                                {'type': 'string'},
                            ],
                        },
                    },
                ],
            },
        },
        'additionalProperties': False,
    }

    CHECKS_JSON_SCHEMA = {
        'check_definitions': InlineChecker.definitions(),
        'type': 'object',
        'patternProperties': {
            '\d+$': {
                '$ref': '#/check_definitions/BOOL_VALUE_OPERATORS',
            },
        },
        'additionalProperties': False,
    }
    INPUTS = [
        {
            'type': 'object',
            'properties': {
                'type': {
                    'enum': ['choice'],
                },
                'group': {
                    'type': 'integer',
                },
                'options': {
                    'type': 'object',
                    'properties': {
                        'choices': {
                            'type': 'array',
                            'items': {
                                'type': 'string',
                            },
                        },
                        'dropdown': {
                            'type': 'boolean',
                        },
                    },
                    'required': ['choices'],
                    'additionalProperties': False,
                },
            },
            'required': ['type', 'group', 'options'],
            'additionalProperties': False,
        },
        {
            'type': 'object',
            'properties': {
                'type': {
                    'enum': ['field'],
                },
                'group': {
                    'type': 'integer',
                },
                'options': {
                    'type': 'object',
                    'properties': {
                        'type_content': {
                            'enum': ['text', 'number', 'strict', 'spaceless'],
                        },
                        'width': {
                            'type': 'integer',
                            'minimum': 1,
                        },
                        'disable_formatting': {
                            'type': 'boolean',
                        },
                    },
                    'required': ['type_content'],
                    'additionalProperties': False,
                },
            },
            'required': ['type', 'group', 'options'],
            'additionalProperties': False,
        },
        {
            'type': 'object',
            'properties': {
                'type': {
                    'enum': ['comma'],
                },
                'group': {
                    'type': 'integer',
                },
                'options': {
                    'type': 'object',
                    'properties': {},
                    'additionalProperties': False,
                }
            },
            'required': ['type', 'group'],
            'additionalProperties': False,
        },
        {
            'type': 'object',
            'properties': {
                'type': {
                    'enum': ['separator'],
                },
                'group': {
                    'type': 'integer',
                },
                'options': {
                    'type': 'object',
                    'properties': {
                        'choices': {
                            'type': 'array',
                            'items': {
                                'enum': ['separate', 'hyphen', 'together'],
                            }
                        },
                    },
                    'required': ['choices'],
                    'additionalProperties': False,
                },
            },
            'required': ['type', 'group', 'options'],
            'additionalProperties': False,
        },
        {
            'type': 'object',
            'properties': {
                'type': {
                    'enum': ['rational'],
                },
                'group': {
                    'type': 'integer',
                },
                'options': {
                    'type': 'object',
                    'properties': {
                        'width': {
                            'type': 'integer',
                            'minimum': 1,
                        },
                    },
                    'additionalProperties': False,
                },
            },
            'required': ['type', 'group', 'options'],
            'additionalProperties': False,
        },
    ]
    JSON_SCHEMA_OPTIONS = {
        'type': 'object',
        'properties': {
            'text': {
                'type': 'string',
            },
            'inputs': {
                'type': 'object',
                'patternProperties': {
                    '\d+$': {
                        'oneOf': INPUTS,
                    },
                },
                'additionalProperties': False,
            },
        },
        'required': ['text', 'inputs'],
        'additionalProperties': False,
    }

    def check(self, user_answer):
        """
        Проводит проверку. При правильности возвращает 0 ошибок, при хотя бы
        одной ошибке - 1
        """
        if self.is_skipped(user_answer):
            return self.SKIPPED, 1
        answer_status = InlineChecker(
            self.data['options']['inputs'],
            self.checks,
            user_answer
        ).check()
        mistakes = int(not all(answer_status.values()))
        return answer_status, mistakes

    @property
    def max_mistakes(self):
        """
        Максимум 1 ошибка
        """
        return 1

    def get_embedded_objects(self):
        """
        Ищем ресурсы и формулы в поле `options.text`,
        """
        return RESOURCE_FORMULA_RE.findall(
            self.data['options']['text'])
