from kelvin.problems.constants import NAMING_TYPES_COMMON, RESOURCE_FORMULA_RE
from kelvin.problems.markers import BaseMarker


class MacaroniMarker(BaseMarker):
    """
    Маркер макарон (соединение двух списков)
    """
    TYPE_NAME = 'macaroni'
    NAMING_TYPES = NAMING_TYPES_COMMON
    ANSWER_JSON_SCHEMA = {
        'oneOf': [
            {'enum': [BaseMarker.SKIPPED_ANSWER]},
            {
                'type': 'array',
                'items': {
                    'type': 'array',
                    'items': {
                        'type': 'integer',
                    },
                    'minItems': 2,
                    'maxItems': 2,
                },
            },
        ],
    }
    RIGHT_ANSWER_JSON_SCHEMA = {
        'type': 'array',
        'items': {
            'type': 'array',
            'items': {
                'type': 'array',
                'items': {
                    'type': 'integer',
                },
                'minItems': 2,
                'maxItems': 2,
            },
            'minItems': 1,
        },
        'minItems': 1,
    }
    JSON_SCHEMA_OPTIONS = {
        'type': 'object',
        'properties': {
            'left': {
                'type': 'object',
                'properties': {
                    'naming': {
                        'enum': NAMING_TYPES,
                    },
                    'hidden': {
                        'type': 'boolean',
                    },
                    'choices': {
                        'type': 'array',
                        'items': {
                            'type': 'string',
                        },
                    },
                    'title': {
                        'type': 'string',
                    },
                },
                'additionalProperties': False,
                'required': ['choices', 'naming'],
            },
            'right': {
                'type': 'object',
                'properties': {
                    'naming': {
                        'enum': NAMING_TYPES,
                    },
                    'hidden': {
                        'type': 'boolean',
                    },
                    'choices': {
                        'type': 'array',
                        'items': {
                            'type': 'string',
                        },
                    },
                    'title': {
                        'type': 'string',
                    },
                },
                'additionalProperties': False,
                'required': ['choices', 'naming'],
            },
        },
        'additionalProperties': False,
        'required': ['left', 'right'],
    }

    def check(self, user_answer):
        """
        Считаем количество ошибок как 0, если ответ полностью совпадает с одним
        из правильных ответов или как количество связей, которые присутствуют
        в правильном ответе и не присутствуют в пользовательском ответе плюс
        количество связей, которые присутствуют в пользовательском ответе,
        но не присутствуют в правильном. Сравниваем с первым ответом у которого
        минимальное количество ошибок.

        Возвращает tuple (статус ответа, количество ошибок)
        В качестве статуса ответа возвращается объект {
            "edges_status": [],
            "missing_edges": [],
            "compare_with_answer": 0,
        }
        edges_status - статус по пользовательскому ответу. Для каждого ребра 1,
         если это ребро присутствует в правильном ответ, 0 - в противном случае
        missing_edges - массив ребер, которые присутствуют в правильном ответе,
         но отсутствуют в пользовательском
        compare_with_answer - порядковый номер ответа, с которым сравниваем
         (нумерация с нуля)

        Количество ошибок всегда будет равно количеству нулей в edges_status +
         длине массива missing_edges
        Для правильного ответа возвращается только поле compare_with_answer

        Пример:
        self.answer = [
            [[0, 1], [1, 2]],
            [[0, 1], [1, 2], [0, 2]]
        ]

        для user_answer = [[0, 1], [0, 2], [1, 2]]
        вернёт ({"compare_with_answer": 1}, 0)

        для user_answer = [[0, 1], [0, 2], [1, 2], [1, 1]]
        вернёт ({
            "edges_status": [1, 0, 1, 0],
            "missing_edges": [],
            "compare_with_answer": 0,
        }, 2)

        :param user_answer: пользовательский ответ (массив массивов из двух
            элементов)
        :return: статус ответа, количество ошибок
        """

        # Если пользователь пропустил задание, то количество ошибок равно длине
        # первого правильного ответа
        if self.is_skipped(user_answer):
            return self.SKIPPED, len(self.answer[0])

        user_answer = [tuple(edge) for edge in user_answer]
        optimal_mistakes_count = float('inf')
        optimal_answer_number = 0

        for answer_num, correct_answer in enumerate(self.answer):

            correct_answer = {tuple(edge) for edge in correct_answer}

            # Если ответы совпадают - сразу вернем, что он правильный
            if set(user_answer) == correct_answer:
                return {'compare_with_answer': answer_num}, 0

            # Считаем количество ошибок
            current_mistakes_count = len(set(user_answer) ^ correct_answer)
            if current_mistakes_count < optimal_mistakes_count:
                optimal_mistakes_count = current_mistakes_count
                optimal_answer_number = answer_num

        optimal_correct_answer = {tuple(edge) for edge in
                                  self.answer[optimal_answer_number]}

        return {
            'edges_status': [int(edge in optimal_correct_answer)
                             for edge in user_answer],
            'missing_edges': [list(edge) for edge in optimal_correct_answer
                              if edge not in user_answer],
            'compare_with_answer': optimal_answer_number,
        }, optimal_mistakes_count

    def get_embedded_objects(self):
        """
        Ищем ресурсы и формулы в полях `options.left.choices` и
        `options.right.choices`
        """

        objects = []
        for options_key in ('left', 'right'):
            choices = self.data['options'].get(options_key, {}).get(
                'choices', [])
            for choice in choices:
                objects.extend(RESOURCE_FORMULA_RE.findall(choice))
        return objects

    @property
    def max_mistakes(self):
        """
        Максимальное число ошибок - возвращаем всегда 1
        """
        # TODO: Сделать подсчет нормальным
        return 1
