from kelvin.problems.constants import NAMING_TYPES_COMMON, RESOURCE_FORMULA_RE
from kelvin.problems.markers import BaseMarker


class MatchingMarker(BaseMarker):
    """
    Маркер сопоставления
    """
    TYPE_NAME = 'matching'
    NAMING_TYPES = NAMING_TYPES_COMMON + ['']
    ANSWER_JSON_SCHEMA = {
        'oneOf': [
            {'enum': [BaseMarker.SKIPPED_ANSWER]},
            {'type': 'object'},
        ],
    }
    RIGHT_ANSWER_JSON_SCHEMA = ANSWER_JSON_SCHEMA
    JSON_SCHEMA_OPTIONS = {
        'type': 'object',
        'properties': {
            'keys': {
                'type': 'object',
                'properties': {
                    'title': {
                        'type': 'string',
                    },
                    'naming': {
                        'enum': NAMING_TYPES,
                    },
                    'choices': {
                        'type': 'array',
                        'items': {
                            'type': 'string',
                        },
                    },
                    'hidden': {
                        'type': 'boolean',
                    },
                },
                'additionalProperties': False,
                'required': ['choices', 'naming'],
            },
            'values': {
                'type': 'object',
                'properties': {
                    'title': {
                        'type': 'string',
                    },
                    'naming': {
                        'enum': NAMING_TYPES,
                    },
                    'choices': {
                        'type': 'array',
                        'items': {
                            'type': 'string',
                        },
                    },
                    'hidden': {
                        'type': 'boolean',
                    },
                },
                'additionalProperties': False,
                'required': ['choices', 'naming'],
            },
            'type_display': {
                'enum': [
                    None,
                    'column',
                    'row',
                ],
            },
        },
        'additionalProperties': False,
        'required': ['keys', 'values'],
    }

    def check(self, user_answer):
        """
        Считаем количество ошибок по количеству правильных связей, которые не
        указаны в ответе пользователя.
        Для статуса возвращает словарь с проставленной корректностью ответов
        Пример:
        для user_answer = {u'А': ['A'], u'Б': ['B'], u'В': 'B'}
            self.answer = {u'A': 'A', u'Б': ['A', 'Б'], u'В': 'В'}

        вернёт {u'А': [1], u'Б': [0], u'В': 1}

        :param user_answer: словарь, где в значениях может быть список
        :return: статус ответа, количество ошибок
        """
        # TODO надо сделать, чтобы в правильном ответе можно было указывать
        # только один вариант?
        # сейчас максимальное число ошибок правильно считается только для этого
        # случая (это сделано, чтобы правильно работал вариант подсчета
        # баллов "по ошибкам" и не исправлять сейчас много кода и данных)
        if self.is_skipped(user_answer):
            return (self.SKIPPED,
                    sum(len(value) if isinstance(value, list) else 1
                        for value in self.answer.values()))
        answer_status = {}
        right_answers = self.answer
        mistakes = 0
        for key in set(list(user_answer.keys()) + list(right_answers.keys())):
            item = []
            answers = user_answer.get(key, [])

            # приводим правильный ответ к списку для более простой проверки
            if key not in right_answers:
                # не добавляем в статус неизвестные ключи и не добавляем
                # по ним ошибок
                continue
            key_right_answers = right_answers[key]
            if not isinstance(key_right_answers, list):
                key_right_answers = [key_right_answers]

            current_mistakes = len(key_right_answers)
            if isinstance(answers, list):
                for answer in answers:
                    if answer in key_right_answers:
                        item.append(self.CORRECT)
                        current_mistakes -= 1
                    else:
                        item.append(self.INCORRECT)
                if item:
                    answer_status[key] = item
            else:
                if answers in key_right_answers:
                    item = self.CORRECT
                    current_mistakes -= 1
                else:
                    item = self.INCORRECT
                answer_status[key] = item
            mistakes += current_mistakes
        return answer_status, mistakes

    def get_embedded_objects(self):
        """
        Ищем ресурсы и формулы в полях `options.keys.title`,
        `options.values.title`, `options.keys.choices.value` и
        `options.values.choices.value`
        """
        objects = []
        for options_key in ('keys', 'values'):
            objects.extend(RESOURCE_FORMULA_RE.findall(
                self.data['options'].get(options_key, {}).get('title', '')))
            choices = self.data['options'].get(options_key, {}).get(
                'choices', [])
            for choice in choices:
                objects.extend(RESOURCE_FORMULA_RE.findall(choice))
        return objects

    @property
    def max_mistakes(self):
        """
        Число ошибок - это число ключей, которые можно указать неправильно
        """
        return len(self.data['options']['keys']['choices']) or 1
