import re
from builtins import map, object
from collections import Counter, defaultdict

from django.core.exceptions import ValidationError as DjangoValidationError
from django.utils.functional import cached_property

from kelvin.problems.constants import RESOURCE_FORMULA_RE
from kelvin.problems.markers import BaseMarker


class WordPartsMarker(BaseMarker):
    """
    Маркер разбора слов по составу
    """
    TYPE_NAME = 'wordparts'
    PARTS_SELECTION_SCHEMA = {
        'type': 'array',
        'items': {
            'type': 'array',
            'items': {
                'type': 'integer',
                'minimum': 0,
            },
            'minItems': 2,
            'maxItems': 2,
        }
    }
    ANSWER_JSON_SCHEMA = {
        'type': 'array',
        'items': {
            'type': 'object',
            'properties': {
                'word_id': {'type': 'string'},
                'parts': {
                    'type': 'object',
                    'properties': {
                        'prefixes': PARTS_SELECTION_SCHEMA,
                        'roots': PARTS_SELECTION_SCHEMA,
                        'suffixes': PARTS_SELECTION_SCHEMA,
                        'stems': PARTS_SELECTION_SCHEMA,
                        'endings': PARTS_SELECTION_SCHEMA,
                    },
                    'additionalProperties': False,
                },
            },
            'additionalProperties': False,
            'required': ['word_id', 'parts'],
        },
    }
    RIGHT_ANSWER_JSON_SCHEMA = ANSWER_JSON_SCHEMA
    JSON_SCHEMA_OPTIONS = {
        'type': 'object',
        'properties': {
            'text': {
                'type': 'string',
            },
            'min_word_length': {
                'type': 'integer',
                'minimum': 1,
            },
        },
        'additionalProperties': False,
        'required': ['text', 'min_word_length'],
    }

    class WordCheckStatus(object):
        CORRECT = 'correct'
        INCORRECT = 'incorrect'
        WRONG_WORD = 'wrong_word'
        SKIPPED = 'skipped'

    TEXT_WORD_REGEX = re.compile(u'([a-zа-яё]+)', flags=re.UNICODE)

    def check(self, user_answer):
        """
        Рассматриваем слова, разобранные в верном ответе и ответе пользователя.
        Если в одном из ответов слово не разобрано, ставим соответствующий
        статус. Иначе проверяем, что среди верных разборов слова (берем
        элементы с одним word_id) есть хотя бы один эквивалентный
        пользовательскому.

        Возвращает tuple (статус ответа, количество ошибок)
        В качестве статуса ответа возвращается объект {
            "status": 1,
            "results": [
                {
                    "word_id": "word:1",
                    "status": "correct"
                },
                ...
            ]
        }
        Где:
        status - общий статут ответа. 1 - правильно, 0 - неправильно
        results - массив результатов проверки ответов. Учитываются как слова из
        верного ответа, так и из ответа пользователя. status может принимать
        такие значения:
            correct - разбор слова верный
            incorrect - разбор слова неверный
            wrong_word - было разобрано слово, которое не нужно разбирать
            skipped - слово не было разобрано
        Последние три случая считаются за ошибки.

        :param user_answer: ответ пользователя (массив разборов слов)
        :return: статус ответа, количество ошибок
        """
        if self.is_skipped(user_answer):
            return self.SKIPPED, self.max_mistakes

        right_answers_by_word = self._group_parts_by_word(self.answer)
        user_answer_by_word = {
            answer['word_id']: answer['parts']
            for answer in user_answer
        }

        check_results = []
        errors_count = 0

        words_ids = set(list(right_answers_by_word.keys()) +
                        list(user_answer_by_word.keys()))
        for word_id in words_ids:
            check_result = {'word_id': word_id}

            # слова нет в верном ответе
            if word_id not in right_answers_by_word:
                check_result['result'] = self.WordCheckStatus.WRONG_WORD
                errors_count += 1
            # слова нет в ответе пользователя
            elif word_id not in user_answer_by_word:
                check_result['result'] = self.WordCheckStatus.SKIPPED
                errors_count += 1
            # слово есть в обоих ответах, можно проверять
            else:
                is_correct = self._check_word_parts(
                    right_answers_by_word[word_id],
                    user_answer_by_word[word_id]
                )
                if is_correct:
                    check_result['result'] = self.WordCheckStatus.CORRECT
                else:
                    check_result['result'] = self.WordCheckStatus.INCORRECT
                    errors_count += 1

            check_results.append(check_result)

        check_status = self.INCORRECT if errors_count else self.CORRECT
        status = {
            'status': check_status,
            'results': check_results,
        }

        return status, errors_count

    def validate(self):
        """
        Добавляет проверку корректности идентификаторов слов
        """
        super(WordPartsMarker, self).validate()

        words_counter = Counter(self._text_words)

        for answer in self.answer:
            self._validate_word_id(words_counter, answer['word_id'])

    @property
    def max_mistakes(self):
        """
        Максимальное число ошибок равно количеству слов в тексте
        """
        return len(self._text_words)

    @staticmethod
    def _group_parts_by_word(answers):
        """
        Группирует разборы слов по word_id
        """
        grouped_parts = defaultdict(list)

        for answer in answers:
            grouped_parts[answer['word_id']].append(answer['parts'])
        return grouped_parts

    @classmethod
    def _check_word_parts(cls, correct_parts_list, answer_parts):
        """
        Проверяет, что заданный разбор есть среди верных
        """
        return (cls._convert_to_comparable(answer_parts)
                in list(map(cls._convert_to_comparable, correct_parts_list)))

    @staticmethod
    def _convert_to_comparable(parts):
        """
        Возвращает разбор слова в таком виде, чтобы его можно было
        сравнить с другим
        """
        return {
            key: set(map(tuple, positions))
            for key, positions in parts.items()
            if positions
        }

    def _validate_word_id(self, words_counter, word_id):
        """
        Проверяет, что идентификатор слова соответствует существующему
        слову в тексте
        """
        current_word, idx = word_id.split(':')
        try:
            idx = int(idx)
        except ValueError:
            idx = None

        if not (idx and (1 <= idx <= words_counter[current_word])):
            raise DjangoValidationError(u'Incorrect word_id: {}'
                                        .format(word_id))

    @cached_property
    def _text_words(self):
        """
        Возвращает список слов из текста, подходящих для разбора
        """
        # удаляем ресурсы
        text = RESOURCE_FORMULA_RE.sub(u' ', self.data['options']['text'])
        min_word_length = self.data['options']['min_word_length']

        return [
            word for word in self.TEXT_WORD_REGEX.findall(text.lower())
            if len(word) >= min_word_length
        ]
