"""
    Эта миграция данных создает для всех текстовых ресурсов тип "Теория"
    с определенным изображением.

    На продакшене и тестинге будут использоваться определенные ресурсы
    по захардкоженным ID. Локально будет выбран первый доступный ресурс,
    если его нет, то ресурс по умолчанию.

    see: EDUCATION-332
"""
import logging

from django.db import migrations
from django.conf import settings

from kelvin.problems.models import DEFAULT_CONTENT_TYPE_NAME


logger = logging.getLogger(__name__)


RESOURCE_IDS = {
    'production': 63068,
    'testing': 86495,
    'development': 86495,
}


# noinspection PyPep8Naming
def create_content_type(apps, schema_editor):
    """
    Создадим новый тип содержимого "Теория"

    :returns TextResourceContentType:
    """
    TextResourceContentType = apps.get_model('problems', 'TextResourceContentType')  # noqa
    Resource = apps.get_model('resources', 'Resource')

    resource_id = RESOURCE_IDS.get(settings.ENVTYPE[0])
    resource = None

    # на проде и тестинге используем уже существующие ресурсы
    # если по каким-то причинам они не найдены, то миграцию не фейлим.
    if resource_id:
        logger.info('Searching for existing resource with id=%d', resource_id)
        resource = Resource.objects.filter(pk=resource_id).first()

    # попробуем вставить первый найденный ресурс
    if not resource:
        logger.info('Searching for first available resource')
        resource = Resource.objects.first()

    # если ничего не сработало – создадим ресурс по-умолчанию
    if not resource:
        resource = Resource.objects.create(file='test.jpg')

    obj, created = TextResourceContentType.objects.get_or_create(
        name=DEFAULT_CONTENT_TYPE_NAME,
        resource=resource,
    )

    if not created:
        logger.info('Content type was already created')

    return obj


# noinspection PyPep8Naming
def update_text_resources(apps, schema_editor):
    """
    Создадим новый тип содержимого и
    обновим поле content_type_object у TextResource
    """
    TextResource = apps.get_model('problems', 'TextResource')
    content_type = create_content_type(apps, schema_editor)

    logger.info('Using content type object "%s"', content_type.name)
    count = TextResource.objects.update(content_type_object=content_type)
    logger.info('Entries updated: %d', count)


class Migration(migrations.Migration):

    dependencies = [
        ('problems', '0004_auto_20170901_1501'),
    ]

    operations = [
        migrations.RunPython(update_text_resources, migrations.RunPython.noop),
    ]
