import os
from builtins import object, str
from copy import deepcopy

from sortedm2m.fields import SortedManyToManyField

from django.conf import settings
from django.db import models
from django.utils.encoding import python_2_unicode_compatible
from django.utils.translation import ugettext_lazy as _

from kelvin.common.fields import JSONField
from kelvin.common.model_mixins import AvailableForSupportMixin, TimeStampMixin, UserBlameMixin
from kelvin.common.utils import safe_filename
from kelvin.problem_meta.models import ProblemMeta
from kelvin.problems.constants import ANDROID_MARKER_TYPES
from kelvin.problems.utils import get_markers_from_layout
from kelvin.resources.models import Resource
from kelvin.subjects.models import Subject, Theme

# Этот тип используется при созданиии по-умолчанию
DEFAULT_CONTENT_TYPE_NAME = 'Теория'


def screenshot_upload_path(problem_instance, filename):
    """
    Возвращает путь и имя файла для сохранения скриншота задачи
    """
    return safe_filename(
        filename, os.path.join('screenshot', str(problem_instance.id)))


@python_2_unicode_compatible
class TextResource(TimeStampMixin, models.Model):
    """
    Модель текстового ресурса (шпаргалки, описания задач)
    """
    name = models.CharField(
        verbose_name=_('Название материала'),
        max_length=255,
    )
    owner = models.ForeignKey(
        settings.AUTH_USER_MODEL,
        verbose_name=_('Автор материала'),
    )
    content = models.TextField(
        verbose_name=_('Содержимое материала'),
        blank=True,
    )
    content_type_object = models.ForeignKey(
        'TextResourceContentType',
        verbose_name=_('Тип содержимого с картинками'),
    )
    resources = models.ManyToManyField(
        Resource,
        verbose_name=_('Привязанные ресурсы'),
        blank=True,
    )
    formulas = JSONField(
        verbose_name=_('Формулы'),
        blank=True,  # ?
        default={},  # ?
    )
    subject = models.ForeignKey(
        Subject,
        verbose_name=_('Предмет'),
        blank=True,
        null=True,
    )
    themes = models.ManyToManyField(
        Theme,
        verbose_name=_('Привязанные темы'),
        blank=True,
    )

    class Meta(object):
        verbose_name = _('Текстовый ресурс')
        verbose_name_plural = _('Текстовые ресурсы')

    def __str__(self):
        return self.name


@python_2_unicode_compatible
class TextResourceContentType(models.Model):
    """
    Справочник типов содержимого для текстового ресурса (EDUCATION-332)
    """
    name = models.CharField(
        unique=True,
        verbose_name=_('Название типа содержимого'),
        max_length=255,
    )
    resource = models.ForeignKey(
        Resource,
        verbose_name=_('Изображение'),
    )

    class Meta(object):
        verbose_name = _('Тип содержимого')
        verbose_name_plural = _('Типы содержимого')
        ordering = ['id']

    def __str__(self):
        return self.name

    @classmethod
    def get_default(cls):
        """
        Для сохранения совместимости с API мы используем тип по умолчанию,
        если с фронта тип содержимого не передали.
        """
        return cls.objects.get(name=DEFAULT_CONTENT_TYPE_NAME)


@python_2_unicode_compatible
class Problem(TimeStampMixin, AvailableForSupportMixin, UserBlameMixin, models.Model):
    """
    Задача (упражнение)
    """
    VISIBILITY_PRIVATE = 1
    VISIBILITY_PUBLIC = 2

    PROBLEM_VISIBILITIES = (
        (VISIBILITY_PRIVATE, _('Только мне')),
        (VISIBILITY_PUBLIC, _('Всем')),
    )

    MAX_POINTS_S = 10
    MAX_POINTS_M = 30
    MAX_POINTS_L = 50
    MAX_POINTS_SCORM = 100
    DEFAULT_MAX_POINTS = {
        ProblemMeta.DIFFICULTY_LIGHT: MAX_POINTS_S,
        ProblemMeta.DIFFICULTY_MEDIUM: MAX_POINTS_M,
        ProblemMeta.DIFFICULTY_HARD: MAX_POINTS_L,
    }

    markup = JSONField(
        verbose_name=_('Разметка задачи'),
        # dump_kwargs={'ensure_ascii': False},
    )
    old_markup = JSONField(
        verbose_name=_('Устаревшая разметка задачи'),
        # dump_kwargs={'ensure_ascii': False},
        blank=True,
        null=True,
    )
    converted = models.BooleanField(
        verbose_name=_('Флаг конвертированности задачи'),
        blank=True,
        default=False,
    )
    subject = models.ForeignKey(
        Subject,
        verbose_name=_('Учебный предмет задачи'),
    )
    name = models.CharField(
        verbose_name=_('Название задачи'),
        max_length=255,
        null=True,
        blank=True,
    )
    owner = models.ForeignKey(
        settings.AUTH_USER_MODEL,
        verbose_name=_('Автор задачи'),
    )
    meta = models.ForeignKey(
        ProblemMeta,
        verbose_name=_('Метаинформация о задаче'),
        blank=True,
        null=True,
    )
    custom_answer = models.BooleanField(
        verbose_name=_('Задание с ручной проверкой'),
        default=False,
    )
    resources = models.ManyToManyField(
        Resource,
        verbose_name=_('Ресурсы, используемые в задаче'),
        blank=True,
    )
    visibility = models.IntegerField(
        verbose_name=_('Видимость'),
        choices=PROBLEM_VISIBILITIES,
        default=VISIBILITY_PRIVATE,
    )
    original = models.BooleanField(
        verbose_name=_('Оригинал'),
        default=True,
    )
    tips = SortedManyToManyField(
        TextResource,
        verbose_name=_('Шпаргалки'),
        blank=True,
    )
    screenshot = models.ImageField(
        verbose_name=_('Скриншот'),
        blank=True,
        null=True,
        upload_to=screenshot_upload_path,
    )
    cover = models.ForeignKey(
        Resource,
        verbose_name=_('Обложка задачи'),
        blank=True,
        null=True,
        related_name='cover_for_problems',
    )
    max_points = models.IntegerField(
        verbose_name=_('Максимальный балл за задачу'),
        # default=MAX_POINTS_M,
        default=1,
        blank=True,
    )
    external_data = JSONField(
        default={},
        verbose_name=_('Внешняя информация для интгерации'),
    )
    type = models.CharField(
        default="generic",
        max_length=64,
        verbose_name=_('Тип проблемы'),
        help_text=_("generic/scorm/review")
    )

    class Meta(object):
        verbose_name = _('Задача')
        verbose_name_plural = _('Задачи')
        ordering = ['date_created']

    def __str__(self):
        return 'Задача {0}'.format(self.id)

    @staticmethod
    def is_markup_android_supported(markup):
        """
        Возвращает `True`, если все типы маркеров в разметке поддерижваются
        андроид-устройствами
        """
        markers = get_markers_from_layout(markup['layout'])
        marker_types = {marker['type'] for marker in markers}
        return marker_types.issubset(ANDROID_MARKER_TYPES)

    def get_clones(self):
        """
        Возвращает всех клонов задачи, включая себя, если является клоном
        """
        return Problem.objects.filter(meta__id=self.meta_id, original=False)

    def get_original(self):
        """
        Возвращает оригинал задачи.
        """
        if self.original or self.meta_id is None:
            return self
        else:
            # TODO: уникальность связки meta-original=True
            try:
                return Problem.objects.filter(meta__id=self.meta_id,
                                              original=True)[0]
            except IndexError:
                # если нет оригинала, возвращаем None
                pass

    def _calculate_max_points(self):
        """
        Возвращает значение max_points на основе сложности задания,
        если у задания есть meta, иначе возвращает значение по умолчанию
        """
        if self.meta_id is None:
            return self.MAX_POINTS_M
        return self.DEFAULT_MAX_POINTS[self.meta.difficulty]

    def save(self, *args, **kwargs):
        """
        Также перед первым сохранением, если `old_markup` пуст и основная
        разметка поддерживается андроид-устройством, заполняем старую разметку
        основной
        """
        if (self.pk is None and
                self.markup and
                not self.old_markup and
                self.is_markup_android_supported(self.markup)):
            self.old_markup = deepcopy(self.markup)

        # Check if problem type is `scorm` and update max_points
        if self.pk is None and self.markup:
            layout = self.markup.get('layout', [])
            for item in layout:
                problem_type = item.get('content', {}).get('type', 'generic')
                self.type = problem_type
                if problem_type == 'scorm':
                    self.max_points = self.MAX_POINTS_SCORM
                    self.problem_type = "scorm"

        # self.edu_id = self.edu_id or None
        # self.max_points = self._calculate_max_points()
        return super(Problem, self).save(*args, **kwargs)


@python_2_unicode_compatible
class ProblemHistory(TimeStampMixin, models.Model):
    """
    Существовавшие версии разметок
    """
    problem = models.ForeignKey(
        Problem,
        verbose_name=_('Задача')
    )
    markup = JSONField(
        verbose_name=_('Разметка задачи'),
        blank=True,
        null=True,
        # dump_kwargs={'ensure_ascii': False},
    )
    old_markup = JSONField(
        verbose_name=_('Устаревшая разметка задачи'),
        # dump_kwargs={'ensure_ascii': False},
        blank=True,
        null=True,
    )
    author = models.ForeignKey(
        settings.AUTH_USER_MODEL,
        verbose_name=_('Автор версии'),
        blank=True,
        null=True,
    )
    message = models.CharField(
        verbose_name=_('Сообщение'),
        max_length=128,
        blank=True,
    )

    class Meta(object):
        verbose_name = _('Версия разметки задачи')
        verbose_name_plural = _('Версии разметок задач')

    def __str__(self):
        return 'Версия задачи {0} от {1}, {2}'.format(
            self.problem_id, self.date_created, self.author)

    @staticmethod
    def add_problem_version(problem, author, message=''):
        """Добавляет версию задачи"""
        ProblemHistory.objects.create(
            problem=problem,
            markup=problem.markup,
            old_markup=problem.old_markup,
            author=author,
            message=message,
        )
