from builtins import object
from datetime import datetime

from django.conf import settings
from django.core.exceptions import ValidationError as DjangoValidationError

from rest_framework import serializers

from kelvin.accounts.serializers import ProblemTheoryOwnerSerializer
from kelvin.common.serializer_fields import JSONDictField, JSONField, NestedForeignKeyMixin
from kelvin.common.serializer_mixins import (
    DateUpdatedFieldMixin, ExcludeForStudentMixin, SerializerForeignKeyMixin, SkipNullsMixin,
)
from kelvin.problem_meta.serializers import ProblemMetaExpansionAwareSerializer
from kelvin.problems.answers import Answer, CustomAnswer
from kelvin.problems.markers import Marker
from kelvin.problems.models import Problem, ProblemHistory, TextResource, TextResourceContentType
from kelvin.problems.validators import ProblemValidators, TextResourceValidators
from kelvin.resources.serializers import ResourceSerializerCorp as ResourceSerializer
from kelvin.subjects.models import Subject, Theme, get_default_subject


class ContentTypeSerializer(NestedForeignKeyMixin,
                            serializers.ModelSerializer):
    """
    Сериализатор модели типов содержимого для текстовых ресурсов
    """
    resource = ResourceSerializer()

    class Meta(object):
        model = TextResourceContentType
        fields = (
            'id',
            'name',
            'resource',
        )


class TextResourceSerializer(SkipNullsMixin,
                             DateUpdatedFieldMixin,
                             ExcludeForStudentMixin,
                             serializers.ModelSerializer):
    """
    Сериализатор модели текстовых ресурсов
    """
    themes = serializers.PrimaryKeyRelatedField(
        queryset=Theme.objects.all(),
        required=False,
        many=True,
    )
    resources = ResourceSerializer(
        read_only=True,
        many=True,
    )
    formulas = JSONDictField(required=False, default={})
    owner = serializers.PrimaryKeyRelatedField(
        required=False,
        read_only=True,
        default=serializers.CreateOnlyDefault(
            serializers.CurrentUserDefault()),
    )
    content_type_object = ContentTypeSerializer(
        required=True,
    )

    class Meta(object):
        model = TextResource
        fields = (
            'id',
            'date_updated',
            'name',
            'content',
            'content_type_object',
            'resources',
            'formulas',
            'themes',
            'owner',
        )
        exclude_for_student = {
            'themes',
        }

    def validate(self, data):
        """
        Проверяет:
            1. Что все указанные тегами в поле `content` ресурсы существуют;
            привязывает их к текстовому ресурсу.
            2. Что формулы в 'content' соответствуют формулам в 'formulas'
        """
        try:
            resource_ids = TextResourceValidators.validate_content(
                data['content'])
            TextResourceValidators.validate_formula_set(
                data['content'], data['formulas'])
        except DjangoValidationError as e:
            raise serializers.ValidationError({'content': e.message})

        data['resources'] = resource_ids
        return data


class TextResourceWithExpandedOwnerSerializer(TextResourceSerializer):
    """
    Сериализатор модели текстовых ресурсов c развёрнутым полем "владелец"
    """
    owner = ProblemTheoryOwnerSerializer(read_only=True)

    class Meta(object):
        model = TextResource
        fields = (
            'id',
            'date_updated',
            'name',
            'content',
            'content_type_object',
            'resources',
            'formulas',
            'themes',
            'owner',
        )


class TextResourceIdDateSerializer(DateUpdatedFieldMixin,
                                   serializers.ModelSerializer):
    """
    Сериализатор текстового ресурса в коротком виде
    """
    class Meta(object):
        model = TextResource
        fields = (
            'id',
            'date_updated',
        )


class ProblemIdDateSerializer(DateUpdatedFieldMixin,
                              serializers.ModelSerializer):
    """
    Сериализатор модели задачи для случаев, когда нужно представить ее
    в коротком виде (id и время обновления)
    """
    class Meta(object):
        model = Problem
        fields = (
            'id',
            'date_updated',
        )


class ProblemSerializer(SkipNullsMixin,
                        DateUpdatedFieldMixin,
                        SerializerForeignKeyMixin,
                        ExcludeForStudentMixin,
                        serializers.ModelSerializer):
    """
    Сериализатор задачи
    """
    default_error_messages = {
        'resources not found': 'resource ids not found: {resource_ids}',
    }

    meta = ProblemMetaExpansionAwareSerializer(required=False)
    resources = ResourceSerializer(read_only=True, many=True)
    subject = serializers.CharField(source='subject.slug', required=False,
                                    default=get_default_subject)
    owner = serializers.PrimaryKeyRelatedField(
        required=False,
        read_only=True,
        default=serializers.CreateOnlyDefault(
            serializers.CurrentUserDefault()),
    )

    fk_update_fields = ['meta']

    class Meta(object):
        model = Problem
        fields = (
            'id',
            'date_updated',
            'meta',
            'markup',
            'owner',
            'visibility',
            'resources',
            'screenshot',
            'max_points',
            'subject',
            'custom_answer',
        )
        read_only_fields = (
            'owner',
            'date_updated',
            'resources',
        )
        exclude_for_student = {
            'subject',
            'meta',
        }

    def validate(self, attrs):
        """
        Проверяет поле `markup` и записывает проверенные ресурсы,
        заполняет поле предмета, если он не дефолтный.
        Если у изменяемой задачи совпадали `markup` и `old_markup`, то
        изменения `markup` также будут записаны в `old_markup`
        """
        if 'markup' in attrs:
            try:
                # проверяем разметку вопросов
                ProblemValidators.validate_markup_json(attrs['markup'])
                attrs['resources'] = ProblemValidators.validate_markup(
                    attrs['markup'])
            except DjangoValidationError as e:
                raise serializers.ValidationError({'markup': e.messages})

            if self.instance and self.instance.markup == self.instance.old_markup:
                attrs['old_markup'] = attrs['markup']

        if 'subject' not in attrs:
            return attrs

        attrs['subject'] = attrs['subject']['slug']
        if not isinstance(attrs['subject'], Subject):
            try:
                attrs['subject'] = Subject.objects.get(slug=attrs['subject'])
            except (TypeError, ValueError, Subject.DoesNotExist):
                raise serializers.ValidationError(
                    {'subject': 'does not exist'})
        return attrs

    def to_representation(self, instance):
        """
        1. скрываем правильные ответы
        2. скрываем поле разметки `cm_comment` для не_контент-менеджеров
        """
        data = super(ProblemSerializer, self).to_representation(instance)

        if self.context.get('hide_answers'):
            data['markup'].pop('solution', None)
            data['markup'].pop('public_solution', None)
            data['markup'].pop('answers', None)
            data['markup'].pop('checks', None)

        if 'request' in self.context:
            user = self.context['request'].user
            if not (user.is_authenticated and user.is_content_manager):
                data['markup'].pop('cm_comment', None)

        return data

    def save(self, **kwargs):
        """
        Создает версию задачи
        """
        instance = super(ProblemSerializer, self).save(**kwargs)
        ProblemHistory.add_problem_version(
            instance, self.context['request'].user, u'Изменения через API')
        return instance


class QuizProblemSerializer(ProblemSerializer):
    """
    Ограниченный набор полей для соревнований
    """
    class Meta(object):
        model = Problem
        fields = (
            'id',
            'markup',
            'resources',
            'max_points',
        )
        read_only_fields = (
            'id',
            'markup',
            'resources',
            'max_points',
        )


class ProblemWithExpandedOwnerSerializer(ProblemSerializer):
    """
    Сериализатор для раскрытия поля `owner`
    """
    owner = ProblemTheoryOwnerSerializer(read_only=True)

    class Meta(object):
        model = Problem
        fields = (
            'id',
            'date_updated',
            'meta',
            'markup',
            'owner',
            'visibility',
            'resources',
            'screenshot',
            'max_points',
            'subject',
            'custom_answer',
        )
        read_only_fields = (
            'date_updated',
            'resources',
            'owner',
        )


class AnswerDatabaseSerializer(serializers.Serializer):
    """
    Сериализатор записи ответа в базу данных
    """
    completed = serializers.BooleanField()
    markers = JSONDictField(allow_null=True)
    theory = JSONField(allow_null=True, required=False)
    custom_answer = JSONField(allow_null=True, required=False)
    max_mistakes = serializers.IntegerField(allow_null=True)
    mistakes = serializers.IntegerField(allow_null=True)
    spent_time = serializers.IntegerField(required=False, allow_null=True)
    points = serializers.IntegerField(required=False, allow_null=True)
    checked_points = serializers.IntegerField(required=False, allow_null=True)
    comment = serializers.CharField(required=False, allow_blank=True)
    answered = serializers.BooleanField(required=False, default=False)

    def create(self, validated_data):
        """
        Создает ответ из валидированных данных
        """
        return Answer(**validated_data)


class UserInCustomAnswerSerializer(serializers.Serializer):
    """
    Сериализатор пользователя в сообщении `custom_answer`
    """
    id = serializers.IntegerField(required=False)

    default_error_messages = {
        'no user_id in context': u'В контексте нет id пользователя',
    }

    def validate(self, attrs):
        """
        Проставляем id пользователя из контекста если он не был указан
        """
        if 'id' not in attrs:
            if self.context and self.context['request'].user:
                attrs['id'] = self.context['request'].user.id
            else:
                self.fail('no user_id in context')
        return attrs


class FileInCustomAnswerSerializer(serializers.Serializer):
    """
    Сериализатор файла в сообщении `custom_answer`
    """
    public_key = serializers.CharField(max_length=255)


class CustomAnswerSerializer(serializers.Serializer):
    """
    Сериализатор сообщения в `custom_answer`
    """
    type = serializers.ChoiceField(choices=CustomAnswer.Type.AVAILABLE_TYPES)
    message = serializers.CharField(allow_blank=True, max_length=2047)
    points = serializers.IntegerField(required=False)
    status = serializers.IntegerField(allow_null=True, required=False)
    date = serializers.DateTimeField(
        required=False,
        default=lambda: datetime.now().strftime(settings.DATETIME_FORMAT),
    )
    user = UserInCustomAnswerSerializer(required=False)
    files = FileInCustomAnswerSerializer(required=False, many=True)

    default_error_messages = {
        'bad type': u'Type of CustomAnswer may be one of {available_types}',
        'bad status': u'Неверный статус',
        'no rules for points': u'Данный тип сообщения не поддерживает '
                               u'выставление баллов',
        'points required': u'В сообщении данного типа обязательно должны быть '
                           u'проставленны баллы',
    }

    def validate(self, attrs):
        """
        Разные проверки
        """
        # Если у сообщения нет прав выставлять оценку, а оценка есть,
        # фейлим валидацию
        if (attrs['type'] not in CustomAnswer.Type.TYPES_WITH_POINTS and
                'points' in attrs):
            self.fail('no rules for points')

        # Если у сообщения нет статуса, но есть баллы за ручную проверку, то
        # фейлим валидацию
        if (attrs.get('status') not in
                CustomAnswer.AVAILABLE_STATUSES_FOR_CHECK and
                'points' in attrs):
            self.fail('bad status')

        # Если у сообщения тип, который подразумевает обязательную оценку, а
        # оценки нет - фейлим валидацию
        if (attrs['type'] in CustomAnswer.Type.TYPES_WITH_POINTS and
                'points' not in attrs):
            self.fail('points required')

        # Заполняем поле user дефолтными значениемя, если оно пусто
        if 'user' not in attrs:
            user = UserInCustomAnswerSerializer(context=self.context, data={})
            user.is_valid()
            attrs['user'] = user.data

        return attrs


class AnswerSerializer(serializers.Serializer):
    """
    Сериализатор ответа
    """
    completed = serializers.BooleanField(required=False)
    markers = JSONField(required=False)
    theory = JSONField(required=False)
    custom_answer = CustomAnswerSerializer(required=False, many=True,
                                           allow_null=True)
    max_mistakes = serializers.IntegerField(required=False, write_only=True)
    mistakes = serializers.IntegerField(required=False, write_only=True)
    spent_time = serializers.IntegerField(required=False, allow_null=True)
    status = serializers.SerializerMethodField()
    points = serializers.SerializerMethodField()
    comment = serializers.CharField(required=False, allow_blank=True,
                                    default='')
    answered = serializers.BooleanField(required=False, default=False)

    default_error_messages = {
        'wrong answer': u'markers, mistakes or spent_time fields '
                        u'should be in answer: '
                        u'{answer}',
        'empty custom answer': u'Для этой задачи обязателен ручной ответ',
    }

    def to_representation(self, instance):
        """
        Пока не нужна информация об ошибках внутри маркеров

        Не показываем проверку, если надо скрывать ответы
        """
        ret = super(AnswerSerializer, self).to_representation(instance)
        if ret.get('markers') is None:
            # неподдерживаемые типы вопросов
            if self.context.get('hide_answers'):
                ret.pop('status')
                ret.pop('points', None)
        else:
            # нормальные вопросы
            if self.context.get('hide_answers'):
                ret.pop('status', None)
                ret.pop('points', None)
                ret.pop('comment', None)
                for marker in ret['markers'].values():
                    marker.pop('answer_status', None)
                    marker.pop('mistakes', None)
                    marker.pop('max_mistakes', None)
            else:
                for marker in ret['markers'].values():
                    if 'mistakes' in marker:
                        mistakes = marker.get('mistakes')
                        if mistakes is None:
                            marker['status'] = Marker.UNCHECKED
                        elif mistakes == 0:
                            marker['status'] = Marker.CORRECT
                        else:
                            marker['status'] = Marker.INCORRECT
                    else:
                        marker['status'] = Marker.INCORRECT
        return ret

    def get_status(self, answer):
        """
        Статус ответа, посчитанный по ошибкам последней попытки
        """
        if (type(answer.custom_answer) is list and
                len(answer.custom_answer)):
            # Задача с ручной проверкой
            last_custom_answer = answer.custom_answer[-1]
            if 'points' not in last_custom_answer:
                return Answer.UNCHECKED
            elif 'status' in last_custom_answer:
                return last_custom_answer['status']
            else:
                return Answer.INCORRECT
        elif answer.mistakes is None:
            return Answer.UNCHECKED
        return Answer.CORRECT if answer.mistakes == 0 else Answer.INCORRECT

    def get_points(self, answer):
        """Если есть проставленные баллы, то отдаются они"""
        return (answer.points if answer.checked_points is None
                else answer.checked_points)

    def to_internal_value(self, data):
        """
        Если в словаре нет ключа `markers`, считаем, что пришло значение
        только для этого ключа
        """
        return super(AnswerSerializer, self).to_internal_value(
            data if 'markers' in data else {'markers': data})

    def validate(self, attrs):
        """
        Проверяет, что:
          1. либо есть ответы на маркеры,
            либо указано число ошибок,
            либо задача не пропущена, это так если:
              нету маркеров или есть флаг answered: False
              но ученик останавливался на задаче (присутствует spent_time)
          2. если задача с ручным ответом, то дожен быть передан ручной ответ
        """
        is_skipped = (
            (
                attrs.get('markers') is None or
                not attrs.get('answered', False)
            ) and attrs.get('spent_time') is not None
        )
        has_markers = isinstance(attrs.get('markers'), dict)
        has_mistakes = 'mistakes' in attrs and 'max_mistakes' in attrs
        if not is_skipped and not has_markers and not has_mistakes:
            self.fail('wrong answer', answer=attrs)

        problem = self._get_problem_from_context()

        if (
            problem and problem.custom_answer and
            not attrs.get('custom_answer') and
            not is_skipped
        ):
            self.fail('empty custom answer')

        return attrs

    def validate_markers(self, answer_markers):
        """
        Отбрасываем лишние маркеры
        """
        question_markers = self.context.get('markers')

        if answer_markers is None:
            return None

        validated_markers = {}
        for id_, answer in answer_markers.items():
            if id_ in question_markers and isinstance(answer, dict):
                Marker(question_markers[id_], None).validate_answer(
                    answer.get('user_answer'))
                validated_markers[id_] = answer
        return validated_markers

    def create(self, validated_data):
        """
        Создаем ответ из валидированных данных
        """
        return Answer(
            markers=validated_data.get('markers'),
            theory=validated_data.get('theory'),
            custom_answer=validated_data.get('custom_answer'),
            completed=validated_data.get('completed', True),
            mistakes=validated_data.get('mistakes'),
            max_mistakes=validated_data.get('max_mistakes'),
            spent_time=validated_data.get('spent_time'),
            points=validated_data.get('points'),
            comment=validated_data.get('comment'),
            answered=validated_data.get('answered'),
        )

    def _get_problem_from_context(self):
        """
        Получаем задачу из контекста
        """
        if 'lesson_problem_link' in self.context:
            return self.context['lesson_problem_link'].problem

        return self.context.get('problem')
