from builtins import str

from celery.utils.log import get_task_logger

from django.conf import settings
from django.core.paginator import Paginator

from kelvin.celery import app
from kelvin.common.revisor import client as revisor_client
from kelvin.courses.models import Course, CourseLessonLink, CoursePermission
from kelvin.problems.models import Problem

logger = get_task_logger(__name__)


@app.task()
def create_revisor_group(problem_id=None):
    """
    Периодически получает список проблем, для которых надо создать группу в микросервисе "Ревизор"
    У таски два контекста:
    1. problem_id is not None: для только что созданной проблемы создать нужные сущности в revisor и сохранить их
    в external_data этой проблемы. Происходит разово - после первоначального сохранения проблемы с типом review
    2. problem_id is None: пробежаться по всем проблемам с типом review и создать сущности в revisor там, где это
    необходимо
    """
    revisor_problems = Problem.objects.filter(
        type="review",
        external_data__revisor_group_id__isnull=True
    )
    if problem_id:
        logger.info("Try to create groups for new problem {}".format(problem_id))
        revisor_problems.filter(id=problem_id)

    if not revisor_problems:
        logger.info("No need to sync with revisor")
        return

    logger.info("Found {} problems to be sycned with revisor".format(revisor_problems.count()))
    for problem in revisor_problems:
        # FIXME: decompose this giant try/catch as soon as revisor goes to production and becomes stable
        try:
            revisor_group_id = revisor_client.create_group(
                data={},
                name="moebius-{}".format(problem.id)
            ).get('id', None)
            problem.external_data['revisor_group_id'] = revisor_group_id
            problem.save()
            logger.info('Created revisor group {} for kelvin problem {}'.format(
                revisor_group_id,
                problem.id
            ))

            # теперь надо проверить, у каких пользователей курсов, к которым принадлежит данная проблема
            # есть право ревьюить, этих пользователей надо завести в ревизоре и добавить их в вышесозданную группу
            problem_courses_permissions = [
                x for x in (
                    CoursePermission.objects.filter(course__lessons__problems=problem.id)
                    .distinct()
                )
                if x.can_review()
            ]

            logger.info("Found {} kelvin users with review permission for problem {}".format(
                len(problem_courses_permissions),
                problem.id
            ))

            for permission_object in problem_courses_permissions:
                if permission_object.can_review():
                    kelvin_user = permission_object.user
                    revisor_user_id = revisor_client.create_user(
                        username=kelvin_user.username,
                        data={
                            'kelvin_user_id': kelvin_user.id
                        }
                    ).get("id", None)
                    logger.info('Created revisor user {}/{}/{} for kelvin user {}'.format(
                        revisor_user_id,
                        kelvin_user.username,
                        kelvin_user.yauid,
                        kelvin_user.id
                    ))
                    revisor_client.create_group_membership(
                        revisor_group_id=revisor_group_id,
                        revisor_user_id=revisor_user_id,
                    )
                    logger.info('Added revisor user {} to revisor group {}'.format(
                        revisor_user_id,
                        revisor_group_id
                    ))
        except Exception as e:
            logger.error("Cannot sync problem {} with revisor. Exception: {}".format(problem.id, str(e)))
