from builtins import map, object, str

import jsonschema

from django.core.exceptions import ValidationError as DjangoValidationError
from django.utils.text import normalize_newlines

from kelvin.problems.constants import (
    COMMON_MARKUP_SCHEMA, COREPROBLEMS_MARKER_TYPES, FORMULA_RE, LAYOUT_BLOCK_KIND_MARKER, LAYOUT_BLOCK_KIND_TEXT,
    MARKERS_WITH_CHECKS, MARKUP_FIELDS_WITH_LINKED_OBJECTS, RESOURCE_FORMULA_RE, RESOURCE_RE,
    RESOURCE_VALIDATED_MARKER_TYPES,
)
from kelvin.problems.markers import Marker
from kelvin.problems.utils import get_markers_from_layout
from kelvin.resources.models import Resource


class ProblemValidators(object):
    """
    Валидаторы модели задачи
    """
    @staticmethod
    def validate_markup_json(markup):
        """
        Проверяем соответствие значения поля json-схеме
        """
        try:
            jsonschema.validate(markup, COMMON_MARKUP_SCHEMA)
        except jsonschema.ValidationError as e:
            raise DjangoValidationError(e.message)

    @staticmethod
    def validate_markup(markup):
        """
        Проверяет, что формулы в разметке соответствуют формулам в поле
        `markup.formulas`, все ресурсы есть в базе
        """
        resource_ids = set()
        marker_ids = set()
        formula_ids = set()

        def fill_objects(name, id_):
            if name == 'marker':
                marker_ids.add(id_)
            elif name == 'formula':
                formula_ids.add(id_)
            elif name == 'resource':
                resource_ids.add(id_)

        for block in markup['layout']:
            if block['kind'] == LAYOUT_BLOCK_KIND_TEXT:
                text = block['content']['text']
                text = normalize_newlines(text)
                for name, id_ in RESOURCE_FORMULA_RE.findall(text):
                    fill_objects(name, id_)
                block['content']['text'] = text

        for field in MARKUP_FIELDS_WITH_LINKED_OBJECTS:
            if field in markup:
                markup[field] = normalize_newlines(markup[field])
                for name, id_ in RESOURCE_FORMULA_RE.findall(
                        markup[field]):
                    fill_objects(name, id_)

        errors = []
        # проверяем маркеры, дополнительно ищем в маркерах ресурсы и формулы
        markers = get_markers_from_layout(markup['layout'])
        for marker in markers:
            id_ = str(marker['id'])
            answer = markup['answers'][id_]
            checks = markup['checks'].get(id_, None)
            marker = Marker(marker, answer, checks)
            try:
                # можно написать одну большую json-схему
                marker.validate()
            except DjangoValidationError as e:
                errors.append(u'В маркере {0} найдена ошибка: {1}'.format(
                    marker.data['id'], e))
            else:
                for name, id_ in marker.get_embedded_objects():
                    fill_objects(name, id_)

        # проверяем, что есть все ресурсы и указаны все формулы
        markup_formulas = set(markup.get('formulas', {}).keys())
        if markup_formulas != formula_ids:
            errors.append(u'Не найдены формулы в `formulas`: {0},\n'
                          u'лишние формулы в `formulas`: {1}'
                          .format(sorted(list(formula_ids - markup_formulas)),
                                  sorted(list(markup_formulas - formula_ids))))

        resources = Resource.objects.filter(id__in=resource_ids).values_list(
            'id', flat=True)
        if len(resources) != len(resource_ids):
            # в `resource_ids` значения — стрóки
            resources = list(map(str, resources))
            errors.append(u'Не найдены ресурсы с идентификаторами: {0}'
                          .format(sorted(list(resource_ids - set(resources)))))

        # проверяем, что для соответствующих маркеров есть проверки
        for marker in [x for x in markup['layout'] if x['kind'] == LAYOUT_BLOCK_KIND_MARKER]:
            if (marker['content']['type'] in MARKERS_WITH_CHECKS and
                    str(marker['content']['id']) not in markup['checks']):
                errors.append(u'Не найдена проверка для маркера {0}'
                              .format(marker['content']['id']))

        if errors:
            raise DjangoValidationError(errors)
        return resources

    @staticmethod
    def validate_markup_marker_types(markup):
        """
        Проверяет, что все маркеры из тех типов, для которых мы можем проверить
        ресурсы
        """
        marker_ids = []
        for marker in get_markers_from_layout(markup['layout']):
            if marker['type'] not in RESOURCE_VALIDATED_MARKER_TYPES:
                marker_ids.append(str(marker['id']))
        if marker_ids:
            raise DjangoValidationError(
                u'Следующие маркеры имеют непроверяемые типы: ' + u''.join(marker_ids)
            )


class TextResourceValidators(object):
    """
    Валидаторы модели TextResource
    """
    @staticmethod
    def validate_content(content):
        """
        Возвращает айдишники существующих ресурсов,
        указанных тегами в `content`, кидает `DjangoValidationError`,
        если указаны несуществующие ресурсы.
        """
        ids = set()
        for id_ in RESOURCE_RE.findall(content):
            ids.add(int(id_))

        existing = Resource.objects.filter(
            id__in=ids).values_list('id', flat=True)

        if len(ids) != len(existing):
            raise DjangoValidationError('some resources do not exist: {0}'
                                        .format(list(ids - set(existing))))

        return ids

    @staticmethod
    def validate_formula_set(content, formulas):
        """
        Проверяет, что все формулы, указанные в 'content', есть в 'formulas',
        и при этом в 'formulas' лишних нет
        """
        formula_ids = set()
        for id_ in FORMULA_RE.findall(content):
            formula_ids.add(id_)
        content_formula_ids = set(formulas.keys())
        if formula_ids != content_formula_ids:
            raise DjangoValidationError(
                u'Не найдены формулы: {0}, лишние формулы: {1}'
                .format(sorted(list(formula_ids - content_formula_ids)),
                        sorted(list(content_formula_ids - formula_ids)))
            )
