import json

from django_filters.rest_framework import DjangoFilterBackend
from rest_condition.permissions import And, Or

from django.core.exceptions import ValidationError
from django.db.models.query_utils import Q
from django.utils import timezone

from rest_framework import viewsets
from rest_framework.decorators import detail_route, list_route, parser_classes
from rest_framework.filters import OrderingFilter
from rest_framework.parsers import MultiPartParser
from rest_framework.permissions import SAFE_METHODS
from rest_framework.response import Response
from rest_framework.status import HTTP_400_BAD_REQUEST

from kelvin.accounts.permissions import (
    IsContentManager, IsStaff, IsTeacher, ObjectForContentManager, ObjectForOwner, ObjectForStaff, ReadOnly,
)
from kelvin.common.permissions import ObjectDetailRequest
from kelvin.common.utils import ANDROID_CLIENT_NAME
from kelvin.courses.models import CourseLessonLink
from kelvin.courses.permissions import CanEditProblem
from kelvin.lessons.models import LessonProblemLink
from kelvin.problems.answers import CustomAnswer, check_answer
from kelvin.problems.filters import ProblemFilter
from kelvin.problems.markers import Marker
from kelvin.problems.models import Problem, TextResource, TextResourceContentType
from kelvin.problems.serializers import (
    AnswerSerializer, ContentTypeSerializer, ProblemSerializer, ProblemWithExpandedOwnerSerializer,
    TextResourceSerializer, TextResourceWithExpandedOwnerSerializer,
)
from kelvin.problems.utils import get_markers_dict_from_layout


class ProblemViewSet(viewsets.ModelViewSet):
    """
    Получение, добавление, изменение и удаление задач
    """
    serializer_class = ProblemSerializer
    permission_classes = (
        CanEditProblem,
        Or(ReadOnly,
           IsTeacher,
           IsContentManager,
           IsStaff),
        Or(ObjectDetailRequest,
           ObjectForOwner,
           ObjectForContentManager,
           ObjectForStaff),
    )
    filter_class = ProblemFilter
    filter_backends = (
        DjangoFilterBackend,
        OrderingFilter,
    )
    ordering_fields = (
        'date_created',
    )

    def get_serializer_class(self):
        """
        Для андроид-клиентов отдает специальный сериализатор, всем остальным -
        стандартный
        """
        if self.request.META.get('client_application') == ANDROID_CLIENT_NAME:
            return self.serializer_class
        if self.action == 'retrieve':
            return ProblemWithExpandedOwnerSerializer
        return self.serializer_class

    @detail_route(methods=['post'], permission_classes=[
        IsTeacher, IsContentManager,
    ])
    def answer(self, request, **kwargs):
        """
        Принимаем ответ на вопрос и возвращаем проверенный ответ с вопросом.
        На вопрос может ответить любой пользователь, если вопрос ему доступен.
        """
        problem = self.get_object()

        # Добавляем в ответ каркас сообщения ручной проверки, если он в задаче
        # обязателен и его нет в запросе
        data = request.data
        if problem.custom_answer and 'custom_answer' not in data:
            data['custom_answer'] = [{
                'type': CustomAnswer.Type.SOLUTION,
                'message': '',
            }]

        # десериализуем ответ
        answer_serializer = AnswerSerializer(
            data=data,
            context={
                'markers': get_markers_dict_from_layout(
                    problem.markup['layout']),
                'problem': problem,
                'request': request,
            },
        )
        answer_serializer.is_valid(raise_exception=True)
        user_answer = answer_serializer.save()

        # проверяем ответ
        checked_answer = check_answer(problem, user_answer)

        # сериализуем вопрос и проверенный ответ
        question_data = self.get_serializer().to_representation(problem)
        question_data['attempt'] = (
            AnswerSerializer(context=self.get_serializer_context())
            .to_representation(checked_answer)
        )
        return Response(question_data)

    @detail_route(methods=['post'], permission_classes=[IsContentManager])
    @parser_classes((MultiPartParser,))
    def screenshot(self, request, **kwargs):
        """
        Обновляем скриншот задачки
        """
        if 'file' not in request.data:
            return Response({'file': 'field required'},
                            status=HTTP_400_BAD_REQUEST)
        file_ = request.data['file']

        problem = self.get_object()
        problem.screenshot.delete(save=False)
        problem.screenshot.save(file_.name, file_)

        serializer = self.get_serializer(problem)
        return Response(serializer.data)

    def get_serializer_context(self):
        """
        Добавляем параметры из запроса в контекст сериализатора
        """
        ctx = super(ProblemViewSet, self).get_serializer_context()
        ctx['hide_answers'] = self.request.query_params.get('hide_answers')

        # TODO переместить сюда `expand_meta`
        return ctx

    def get_queryset(self):
        """
        Для анонимных пользователям предоставляем доступ к открытым вопросам.
        Для учителя и ученика оставляет только доступные вопросы,
        для редакторов возвращает все, подходящие под фильтры
        """
        user = self.request.user
        if not user.is_authenticated:
            if self.action == 'retrieve':
                return (Problem.objects
                        .filter(visibility=Problem.VISIBILITY_PUBLIC)
                        .select_related('owner'))
            return Problem.objects.filter(visibility=Problem.VISIBILITY_PUBLIC)
        queryset = Problem.objects.all()
        if self.action == 'retrieve':
            queryset = queryset.select_related('owner')

        # TODO перенести проверку параметров фильтрации в ProblemFilter
        course_id = self.request.query_params.get('course')
        lesson_id = self.request.query_params.get('lesson')
        exclude_lesson_id = self.request.query_params.get('exclude_lesson')

        # Список фильтрации по параметрам и правам
        filters = []

        # Ограничивает доступные задач по пользователю
        now = timezone.now()
        if user.is_content_manager:
            pass
        elif user.is_teacher:
            filters.append(
                # clesson-ы в учительском курсе
                Q(course__owner=user,
                  accessible_to_teacher__lt=now) |

                # оригиналы clesson-ов, в учительском курсе
                Q(copies__course__owner=user,
                  copies__accessible_to_teacher__lt=now) |

                # бесплатные курсы
                Q(course__free=True,
                  accessible_to_teacher__lt=now)
            )
        else:
            filters.append(Q(course__students=user))

        # Фильтрация по курсу или уроку
        if course_id:
            filters.append(Q(course_id=course_id))
        if lesson_id:
            filters.append(Q(lesson_id=lesson_id))

        if filters:
            clessons = CourseLessonLink.objects.filter(*filters)
            lesson_ids = clessons.values_list('lesson_id')
            problem_ids = LessonProblemLink.objects.filter(
                lesson_id__in=lesson_ids).values_list('problem_id')
            q_object = Q(id__in=problem_ids)
            if not (course_id or lesson_id):
                # если нет фильтрации по курсу или занятию, то добавляем
                # собственные задачи и доступные всем
                q_object |= (Q(owner=self.request.user) |
                             Q(visibility=Problem.VISIBILITY_PUBLIC))
            queryset = queryset.filter(q_object)

        if exclude_lesson_id:
            exclude_problem_ids = LessonProblemLink.objects.filter(
                lesson_id=exclude_lesson_id,
                problem_id__isnull=False,
            ).values_list('problem_id')
            queryset = queryset.exclude(id__in=exclude_problem_ids)

        if self.request.query_params.get('mine'):
            queryset = queryset.filter(owner=self.request.user)
        elif user.is_content_manager and self.action == 'list':
            queryset = queryset.filter(owner__is_content_manager=True)

        queryset = queryset.select_related('meta').order_by('id')
        if self.request.method in SAFE_METHODS:
            queryset = queryset.prefetch_related('resources')
            if 'expand_meta' in self.request.query_params:
                queryset = (
                    queryset.select_related('meta__main_theme')
                            .prefetch_related(
                                'meta__group_levels',
                                'meta__problemmetaexam_set',
                                'meta__problemmetaexam_set__exam',
                                'meta__skills',
                                'meta__additional_themes',
                    )
                )

        return queryset


class TextResourceViewSet(viewsets.ModelViewSet):
    """
    Получение, добавление, изменение и удаление текстовых ресурсов
    """
    queryset = (
        TextResource.objects.all()
        .select_related('content_type_object', 'content_type_object__resource')
        .prefetch_related('themes', 'resources')
    )
    serializer_class = TextResourceSerializer
    permission_classes = (
        Or(ReadOnly,
           IsTeacher,
           IsContentManager,
           IsStaff),
        Or(ObjectDetailRequest,
           ObjectForOwner,
           ObjectForContentManager,
           ObjectForStaff),
    )

    def get_serializer_class(self):
        """
        Отдаём развёрнутую инфу про текстовый ресурс (свойства юзера),
        если retreive-запрос.
        """
        if self.action == 'retrieve':
            return TextResourceWithExpandedOwnerSerializer
        return self.serializer_class


class ContentTypeViewSet(viewsets.ModelViewSet):
    """
    CRUD представление для типов контента.

    Доступ к ресурсу есть только у пользователя, который
    и контент менеджер и учитель одновременно + staff.
    """
    queryset = TextResourceContentType.objects.select_related('resource')
    serializer_class = ContentTypeSerializer
    permission_classes = (
        Or(IsContentManager, IsTeacher, IsStaff),
    )


class MarkerViewSet(viewsets.GenericViewSet):
    """
    ViewSet для проверки ответа на конкретный маркер
    """

    serializer_class = TextResourceSerializer

    @list_route(methods=['POST'])
    def check_answer(self, request):
        """
        Ручка для проверки произвольного маркера
        """
        marker_data = json.loads(request.data.get('marker_data'))
        user_answer = json.loads(request.data.get('user_answer'))

        marker = Marker(*marker_data)

        try:
            marker.validate()
        except ValidationError as e:
            return Response(status=HTTP_400_BAD_REQUEST)

        return Response(marker.check(user_answer))
