from django.contrib import admin

from kelvin.projects.models import Project, ProjectSubject, ProjectSubjectItem


class ProjectSubjectInline(admin.TabularInline):
    """
    Инлайн модель ПроектоПредмета для отображения в проекте
    """
    model = ProjectSubject
    ordering = ('order',)
    extra = 0


class ProjectSubjectItemInline(admin.TabularInline):
    """
    Инлайн модель связи ПроектоПредмета и объекта
    """
    model = ProjectSubjectItem
    ordering = ('order',)
    raw_id_fields = ('course',)
    extra = 0


class ProjectAdmin(admin.ModelAdmin):
    """
    Модель админки для проекта
    """
    list_display = ('id', 'slug', 'title', 'description', 'default_project_subject', 'add_code')
    inlines = (ProjectSubjectInline, )
    readonly_fields = ('add_code',)

    def get_form(self, request, obj=None, **kwargs):
        """
        Настройка формы для админки
        В качестве дефолтного ПроектоПредмета можно выбрать только тот, который
        относится к этому проекту
        """
        form = super(ProjectAdmin, self).get_form(request, obj, **kwargs)
        default_project_subject_field = form.base_fields.get('default_project_subject', None)
        if default_project_subject_field:
            default_project_subject_field.queryset = (
                ProjectSubject.objects.filter(project=obj)
            )
        return form


class ProjectSubjectAdmin(admin.ModelAdmin):
    """
    Модель админки для ПроектоПредмета
    """
    list_display = ('id', 'project', 'slug', 'title', 'order')
    list_filter = ('project', )
    inlines = (ProjectSubjectItemInline, )
    ordering = ('project', 'order')


class ProjectSubjectItemAdmin(admin.ModelAdmin):
    """
    Модель админки для карточек (связи ПроектоПредмета и контента)
    """
    list_display = ('id', 'project_subject', 'display_type', 'item_type',
                    'order')
    ordering = ('project_subject', 'order')
    raw_id_fields = ('course',)

    def get_form(self, request, obj=None, **kwargs):
        """
        Настройка формы для админки
        Сортирует выводимые ПроектоПредметы и оптимизирует запрос к БД
        """
        form = super(ProjectSubjectItemAdmin, self).get_form(
            request, obj, **kwargs)
        form.base_fields['project_subject'].queryset = (
            ProjectSubject.objects
            .select_related('project')
            .order_by('project', 'order')
        )
        return form


admin.site.register(Project, ProjectAdmin)
admin.site.register(ProjectSubject, ProjectSubjectAdmin)
admin.site.register(ProjectSubjectItem, ProjectSubjectItemAdmin)
