import hashlib
from builtins import object

from django.core.exceptions import ValidationError
from django.db import models
from django.utils.encoding import python_2_unicode_compatible
from django.utils.translation import ugettext_lazy as _

from kelvin.common.fields import JSONField
from kelvin.common.model_mixins import TimeStampMixin
from kelvin.projects.schemas import (
    DisplayOptionsFieldSchema, MetaFieldSchema, display_options_schema_validator, meta_schema_validator,
)
from kelvin.subjects.models import Subject

DEFAULT_PROJECT_SLUG = 'DEFAULT'


class ProjectQuerySet(models.QuerySet):
    def default(self):
        return self.filter(slug=DEFAULT_PROJECT_SLUG).first()


@python_2_unicode_compatible
class Project(TimeStampMixin, models.Model):
    """
    Модель проекта (ege, oge,...)
    """
    slug = models.SlugField(
        verbose_name=_('Слаг-идентификатор проекта'),
        max_length=255,
        unique=True,
    )
    title = models.CharField(
        verbose_name=_('Название проекта'),
        max_length=255,
    )
    description = models.TextField(
        verbose_name=_('Описание проекта'),
        blank=True,
        default=""
    )
    default_project_subject = models.ForeignKey(
        'ProjectSubject',
        verbose_name=_('ПроектоПредмет по умолчанию'),
        related_name='+',
        blank=True,
        null=True,
        on_delete=models.CASCADE,
    )
    background_image = models.ImageField(
        verbose_name=_('Фоновое изображение'),
        blank=True,
        null=True
    )
    add_code = models.CharField(
        verbose_name=_('Уникальный код проекта'),
        help_text=_('Для самодобваления пользователей в проект'),
        max_length=255,
        blank=True,
        null=False,
        unique=True
    )
    nda = models.BooleanField(
        verbose_name=_('NDA'),
        default=False,
    )
    nda_resource = models.ForeignKey(
        'resources.Resource',
        verbose_name=_('Файл с nda'),
        null=True,
        blank=True,
        on_delete=models.SET_NULL,
    )

    default_experiment_flags = JSONField(
        verbose_name=_('Дефолтные пользовательские флаги для экспериментов'),
        blank=False,
        default={},
    )

    objects = ProjectQuerySet.as_manager()

    class Meta(object):
        verbose_name = _('Проект')
        verbose_name_plural = _('Проекты')

    def __str__(self):
        return '{self.id}: [{self.slug}] {self.title}'.format(self=self)

    @classmethod
    def gen_digest(cls, input_data=""):
        gen = hashlib.md5()
        gen.update(str.encode(input_data))
        return gen.hexdigest()

    def save(self, *args, **kwargs):
        if not self.add_code:
            self.add_code = self.gen_digest(input_data="Project {}".format(self.slug))
        super(Project, self).save(*args, **kwargs)


@python_2_unicode_compatible
class ProjectSubject(TimeStampMixin, models.Model):
    """
    Модель ПроектоПредмета
    """
    project = models.ForeignKey(
        Project,
        related_name='subjects',
        verbose_name=_('Проект'),
        on_delete=models.CASCADE,
    )
    subject = models.ForeignKey(
        Subject,
        verbose_name=_('Предмет'),
        on_delete=models.CASCADE,
        null=True,
    )
    slug = models.CharField(
        verbose_name=_('Слаг-идентификатор ПроектоПредмета'),
        help_text=_('Уникальный слаг-идентификатор в рамках одного проекта'),
        max_length=255,
    )
    title = models.CharField(
        verbose_name=_('Название ПроектоПредмета'),
        max_length=255,
    )
    full_title = models.CharField(
        verbose_name=_('Полное название ПроектоПредмета'),
        max_length=511,
        blank=True,
    )
    order = models.PositiveIntegerField(
        verbose_name=_('Порядок сортировки'),
    )
    background = models.CharField(
        verbose_name=_('Фоновый цвет'),
        max_length=10,
    )
    meta = JSONField(
        verbose_name=_('Мета-информация проектопредмета'),
        help_text=_(
            'Объект с мета-информацией о проектопредмете, например:'
            '{}'.format(MetaFieldSchema.EXAMPLE)
        ),
        validators=[meta_schema_validator],
        default={},
        blank=True,
    )

    class Meta(object):
        unique_together = (
            ('project', 'order'),
            ('project', 'slug'),
        )
        verbose_name = _('Предмет в проекте')
        verbose_name_plural = _('Предметы в проекте')

    def __str__(self):
        return ('{self.id}: [{self.project.slug}/{self.slug}] '
                '{self.title}'.format(self=self))


@python_2_unicode_compatible
class ProjectSubjectItem(TimeStampMixin, models.Model):
    """
    Модель связи ПроектоПредмета и наполнения
    """

    class DisplayType(object):
        CARD_NORMAL_FOURTH = 1
        CARD_NORMAL_HALF = 2
        CARD_EXTENDED_FOURTH = 3
        CARD_EXTENDED_HALF = 4

        HAS_SHORT_REPRESENTATION = (
            CARD_NORMAL_FOURTH,
            CARD_NORMAL_HALF,
        )

        choices = (
            (CARD_NORMAL_FOURTH, _('Карточка 1/4')),
            (CARD_NORMAL_HALF, _('Карточка 1/2')),
            (CARD_EXTENDED_FOURTH, _('Расширенная карточка 1/4')),
            (CARD_EXTENDED_HALF, _('Расширенная карточка 1/2')),
        )
        default = CARD_NORMAL_FOURTH

    class ItemType(object):
        COURSE = 1
        LINK = 2

        choices = (
            (COURSE, _('Курс')),
            (LINK, _('Ссылка')),
        )
        default = COURSE

    project_subject = models.ForeignKey(
        ProjectSubject,
        related_name='content_items',
        verbose_name=_('ПроектоПредмет'),
        on_delete=models.CASCADE,
    )
    display_type = models.PositiveSmallIntegerField(
        verbose_name=_('Тип отображения элемента'),
        default=DisplayType.default,
        choices=DisplayType.choices,
    )
    item_type = models.PositiveSmallIntegerField(
        verbose_name=_('Тип контента'),
        default=ItemType.default,
        choices=ItemType.choices,
    )
    course = models.ForeignKey(
        'courses.Course',
        verbose_name=_('Идентификатор курса'),
        on_delete=models.CASCADE,
        blank=True,
        null=True,
    )
    link = models.URLField(
        verbose_name=_('Ссылка'),
        blank=True,
        null=True,
    )
    order = models.PositiveIntegerField(
        verbose_name=_('Порядок сортировки'),
    )
    display_options = JSONField(
        verbose_name=_('Опции отображения элемента'),
        help_text=_(
            'Объект с информацией об отображении элемента, например:'
            '{}'.format(DisplayOptionsFieldSchema.EXAMPLE)
        ),
        validators=[display_options_schema_validator],
        default={},
        blank=True,
    )

    class Meta(object):
        unique_together = ('project_subject', 'order',)
        verbose_name = _('Контент для предмета в проекте')
        verbose_name_plural = _('Элементы контента для предмета в проекте')

    @property
    def has_short_representation(self):
        return self.display_type in self.DisplayType.HAS_SHORT_REPRESENTATION

    def __str__(self):
        if self.item_type == self.ItemType.COURSE:
            return '{self.id}: course-{self.course_id}'.format(self=self)

        elif self.item_type == self.ItemType.LINK:
            return '{self.id}: link-{self.link}'.format(self=self)

        else:
            return '{self.id}'.format(self=self)

    def clean(self):
        """
        Валидация модели на соответствие заполненных полей типу
        и соответствующая чистка данных
        """

        super(ProjectSubjectItem, self).clean()

        # course related validation
        if self.item_type == self.ItemType.COURSE:
            # course type must have course id
            if self.course is None:
                raise ValidationError(_('Выберите связанный курс'))

            # cleaning other type fields
            self.link = None

        # link related validation
        elif self.item_type == self.ItemType.LINK:
            # link type must have link
            if not self.link:
                raise ValidationError(_('Введите ссылку'))

            # cleaning other type fields
            self.course = None
