"""
JSON-схемы для валидации данных.
"""
from builtins import object

import jsonschema

from django.core.exceptions import ValidationError


class MetaFieldSchema(object):
    """
    Класс инкапсулирует логику валидации поля `meta`
    """
    # Схема метаданных для страницы с проекто-предметом (карточками)
    JSON_SCHEMA = {
        'type': 'object',
        'properties': {
            'meta': {
                'type': 'object',
                'properties': {
                    'title': {
                        'type': 'string',
                    },
                    'description': {
                        'type': 'string',
                    },
                    'keywords': {
                        'type': 'string',
                    },
                },
                'additionalProperties': False,
            },
            'social': {
                'type': 'object',
                'properties': {
                    'text': {
                        'type': 'string',
                    },
                    'image': {
                        'type': 'string',
                    }
                },
                'additionalProperties': False,
            },
        },
        'additionalProperties': False,
    }
    # Выводим в подсказке в админке
    EXAMPLE = u"""
<pre>
{
  "meta": {
    "title": "<Название страницы>",
    "description": "<Описание страницы для SEO>",
    "keywords": "<Ключевые слова для SEO через запятую>"
  },
  "social": {
    "text": "<Текст для шаринга в соц. сетях>",
    "image": "<Картинка для шаринга в соц. сетях>"
  }
}
</pre>
"""


def meta_schema_validator(data):
    try:
        jsonschema.validate(data, MetaFieldSchema.JSON_SCHEMA)
    except jsonschema.ValidationError as e:
        raise ValidationError(e.message)


class DisplayOptionsFieldSchema(object):
    """
    Класс инкапсулирует логику валидации поля `display_options`
    """
    # Схема метаданных для страницы с контентом для проекто-предмета
    JSON_SCHEMA = {
        'type': 'object',
        'properties': {
            'content_description': {
                'type': 'string',
            },
            'color': {
                'type': 'string',
            },
            'logo': {
                'type': 'string',
            },
            'background_image': {
                'type': 'string',
            },
            'background_color': {
                'type': 'string',
            }
        },
        'additionalProperties': False,
    }
    # Выводим в подсказке в админке
    EXAMPLE = u"""
<pre>
{
  "content_description": "<Описание типа контента для карточки>",
  "color": "<Цвет текста на карточке в HEX>",
  "logo": "<Ссылка на логотип (например для видеокурсов)>",
  "background_image": "<Ссылка на фон для карточки>",
  "background_color": "<Цвет фона карточки в HEX>"
}
</pre>
    """


def display_options_schema_validator(data):
    try:
        jsonschema.validate(
            data,
            DisplayOptionsFieldSchema.JSON_SCHEMA
        )
    except jsonschema.ValidationError as e:
        raise ValidationError(e.message)
