from builtins import object

from django.db.models import Prefetch

from rest_framework import serializers

from kelvin.accounts.models import UserProject
from kelvin.courses.models import CourseLessonLink
from kelvin.courses.serializers import CourseInProjectExtendedSerializer, CourseInProjectShortSerializer
from kelvin.projects.models import Project, ProjectSubject, ProjectSubjectItem
from kelvin.resources.serializers import ResourceSerializerCorp


class ProjectSubjectShortSerializer(serializers.ModelSerializer):
    """
    Сериализатор модели ПроектоПредмета
    Только основные поля, без вложенных объектов
    """
    class Meta(object):
        model = ProjectSubject
        fields = (
            'id',
            'slug',
            'title',
            'full_title',
            'meta',
            'background',
        )


class ProjectSubjectSerializer(ProjectSubjectShortSerializer):
    """
    Сериализатор модели ПроектоПредмета
    """

    items = serializers.SerializerMethodField()

    class Meta(ProjectSubjectShortSerializer.Meta):
        fields = ProjectSubjectShortSerializer.Meta.fields + ('items', )

    def get_items(self, instance):
        """
        Функция, которая сериализует все элементы внутри ПроектоПредмета
        """

        course_lesson_link_queryset = (
            CourseLessonLink.objects.select_related('lesson')
        )

        course_items = (
            ProjectSubjectItem.objects.filter(
                project_subject=instance,
                item_type=ProjectSubjectItem.ItemType.COURSE,
            )
            .select_related('course')
            .prefetch_related(
                Prefetch('course__courselessonlink_set',
                         queryset=course_lesson_link_queryset)
            )
            .order_by('order')
        )
        return ProjectSubjectItemSerializer(course_items, many=True).data


class ProjectSubjectItemSerializer(serializers.ModelSerializer):
    """
    Сериализатор модели связи ПроектоПредмета и Объекта
    """
    content = serializers.SerializerMethodField()

    class Meta(object):
        model = ProjectSubjectItem
        fields = (
            'id',
            'display_type',
            'order',
            'display_options',
            'content',
        )

    def get_content(self, instance):
        """
        Функция, дополняющая связь ПроектоПредмета с объектом самим объектом
        """
        if instance.item_type == ProjectSubjectItem.ItemType.COURSE:
            if instance.has_short_representation:
                return CourseInProjectShortSerializer(instance.course).data
            else:
                return CourseInProjectExtendedSerializer(instance.course).data


class ProjectSerializerSimple(serializers.ModelSerializer):
    """ Простой сериализатор проекта для отдачи элементов списка """

    nda_resource = ResourceSerializerCorp()

    class Meta(object):
        model = Project
        fields = ('id', 'slug', 'title', 'add_code', 'nda', 'nda_resource')


class ProjectSerializer(serializers.ModelSerializer):
    """
    Сериализатор модели проекта для отдачи информации по отдельному проекту
    """

    default_project_subject_id = serializers.IntegerField()
    project_subjects = serializers.SerializerMethodField()
    nda_resource = ResourceSerializerCorp()

    class Meta(object):
        model = Project
        fields = (
            'id',
            'slug',
            'add_code',
            'title',
            'nda',
            'nda_resource',
            'default_project_subject_id',
            'project_subjects',
        )

    def get_project_subjects(self, instance):
        """
        Функция, дополняющая представление проекта ПроектоПредметами
        В зависимости от параметров url раскрывает дефолтный/запрашиваемый
        ПроектоПредмет
        """
        expand_subject_slug = (self.context['request']
                                   .query_params.get('expand_subject'))

        answer = []
        for project_subject in instance.subjects.all().order_by('order'):
            expand_subject_slug_exist_and_equals = (
                expand_subject_slug and
                expand_subject_slug == project_subject.slug
            )
            no_expand_subject_slug_and_default_equals = (
                not expand_subject_slug and
                project_subject == instance.default_project_subject
            )
            if (expand_subject_slug_exist_and_equals or
                    no_expand_subject_slug_and_default_equals):
                answer.append(ProjectSubjectSerializer(project_subject).data)
            else:
                answer.append(
                    ProjectSubjectShortSerializer(project_subject).data)
        return answer


class UserProjectSerializer(serializers.ModelSerializer):
    # project_nda_resource = ResourceSerializerCorp()
    slug = serializers.ReadOnlyField(source='project.slug')
    title = serializers.ReadOnlyField(source='project.title')
    nda = serializers.ReadOnlyField(source='project.nda')
    id = serializers.ReadOnlyField(source='project.id')
    add_code = serializers.ReadOnlyField(source='project.add_code')
    nda_resource = ResourceSerializerCorp(source='project.nda_resource')

    class Meta(object):
        model = UserProject
        fields = (
            'project',
            'nda_accepted',
            'slug',
            'id',
            'nda',
            'add_code',
            'title',
            'nda_resource',
        )
