from rest_framework import mixins, viewsets
from rest_framework.permissions import IsAdminUser
from rest_framework.response import Response

from kelvin.projects.models import Project, ProjectSubject
from kelvin.projects.serializers import ProjectSerializer, ProjectSerializerSimple, ProjectSubjectSerializer


class ProjectSubjectViewSet(mixins.RetrieveModelMixin,
                            viewsets.GenericViewSet):
    """
    Отображение для ручки
    GET /project_subjects/<id>/
    """
    queryset = ProjectSubject.objects.all()
    serializer_class = ProjectSubjectSerializer


class ProjectViewSet(mixins.RetrieveModelMixin,
                     mixins.ListModelMixin,
                     viewsets.GenericViewSet):
    """
    Отображение для ручки
    GET /projects/ - все проекты
    GET /projects/<project_slug>/[?expand_subject=<project_subject_slug>] - одиночный проект
    """
    queryset = Project.objects.all().select_related('default_project_subject')
    lookup_field = 'slug'
    lookup_url_kwarg = 'pk'
    serializer_class = ProjectSerializer

    def list(self, request):
        """
        Отдаём доступные пользователю проекты.
        Для суперпользователя отдаём все имеющиеся проекты.
        """
        qs = Project.objects.filter(userproject__user=request.user)
        qs = qs.select_related('default_project_subject')

        add_code = request.GET.get('add_code')
        if add_code:
            qs = qs.filter(add_code=add_code)

        serializer = ProjectSerializerSimple(qs, many=True)
        return Response(serializer.data)


class ProjectViewSetAdmin(viewsets.ModelViewSet):
    """ ViewSet for admin only """
    queryset = Project.objects.all().select_related('default_project_subject')
    lookup_field = 'slug'
    lookup_url_kwarg = 'pk'
    serializer_class = ProjectSerializer
    permission_classes = [IsAdminUser]
