from django_object_actions import DjangoObjectActions
from ordered_model.admin import OrderedTabularInline

from django import forms
from django.contrib import admin
from django.urls import reverse
from django.utils.html import format_html
from django.utils.translation import ugettext_lazy as _

from .models import CourseReportRequest, Operation, Query, QueryOutputField, QueryParameter, ResultData
from .services import check_yql_operation, download_yql_result_data, validate_query


class BaseReadOnlyInline(object):
    extra = 0
    can_delete = False
    max_num = 0

    def has_add_permission(self, request):
        return request.user.is_superuser

    def has_delete_permission(self, request, obj=None):
        return False


class ReadOnlyTabularInline(BaseReadOnlyInline, admin.TabularInline):
    pass


class ReadOnlyStackedInline(BaseReadOnlyInline, admin.StackedInline):
    pass


class QueryParameterForm(forms.ModelForm):
    class Meta:
        model = QueryParameter
        fields = '__all__'
        widgets = {
            'help_text': forms.Textarea(attrs={'cols': 50, 'rows': 1})
        }


class QueryParameterInline(OrderedTabularInline):
    model = QueryParameter
    form = QueryParameterForm
    fields = ('name', 'type', 'label', 'help_text', 'default', 'is_hidden', 'order', 'move_up_down_links')
    readonly_fields = ('order', 'move_up_down_links')
    extra = 0
    ordering = ('order',)


class QueryOutputFieldInline(OrderedTabularInline):
    model = QueryOutputField
    fields = ('name', 'title',)
    extra = 0
    ordering = ('order',)


class QueryOperationInline(ReadOnlyTabularInline):
    model = Operation
    fields = ('operation_link', 'action', 'status', 'created', 'modified')
    readonly_fields = fields

    def operation_link(self, obj: Operation):
        return format_html(
            '<a href="{}" target="_blank">{}</a>',
            reverse(
                'admin:reports_operation_change',
                args=(obj.id,)
            ),
            obj.yql_operation_id,
        )
    operation_link.short_description = _("Operation ID")


class QueryCourseReportRequestInline(ReadOnlyTabularInline):
    model = CourseReportRequest
    fields = ('course', 'user', 'operation', 'parameters', 'notify_by_email', 'is_archived', 'created', 'modified')
    readonly_fields = fields


@admin.register(Query)
class QueryAdmin(DjangoObjectActions, admin.ModelAdmin):
    list_display = ('title', 'type', 'version', 'is_active', 'created', 'modified')
    list_filter = ('is_active',)
    search_fields = ('title', 'version')
    raw_id_fields = ('created_by',)

    inlines = [
        QueryParameterInline,
        QueryOutputFieldInline,
        QueryOperationInline,
        QueryCourseReportRequestInline,
    ]

    change_actions = [
        'validate_action',
    ]

    def get_urls(self):
        urls = super().get_urls()
        for inline in self.inlines:
            if hasattr(inline, 'get_urls'):
                urls = inline.get_urls(self) + urls
        return urls

    def validate_action(self, request, obj: Query):
        try:
            validate_query(obj.id)
        finally:
            self.message_user(request, _("Запрос отправлен на валидацию"))

    validate_action.label = _("Валидация")


class OperationCourseReportRequestInline(ReadOnlyTabularInline):
    model = CourseReportRequest
    fields = (
        'course', 'user', 'query', 'parameters', 'format', 'status',
        'notify_by_email', 'is_archived', 'created', 'modified',
    )
    readonly_fields = fields


class OperationResultDataInline(ReadOnlyStackedInline):
    model = ResultData
    fields = (
        'old_file',
        'resource',
        'size',
    )
    readonly_fields = fields


@admin.register(Operation)
class OperationAdmin(DjangoObjectActions, admin.ModelAdmin):
    list_display = ('id', 'yql_operation_id', 'query', 'action', 'status', 'created', 'modified')
    list_filter = ('action', 'status',)
    search_fields = ('yql_operation_id', 'query__title',)
    raw_id_fields = ('created_by',)

    fields = (
        'query',
        'yql_operation_id',
        'action',
        'status',
        'query_data',
        'operation_data',
        'query_title',
        'query_version',
        'errors',
        'issues',
        'created_by',
    )
    readonly_fields = (
        'yql_operation_id',
        'status',
        'query_data',
        'operation_data',
        'query_title',
        'query_version',
        'errors',
        'issues',
    )

    inlines = [
        OperationResultDataInline,
        OperationCourseReportRequestInline,
    ]

    change_actions = [
        'update_status',
        'download_data',
    ]

    def update_status(self, request, obj: Operation):
        try:
            check_yql_operation(obj.id)
        finally:
            self.message_user(request, _("Статус операции '%s' был обновлен") % str(obj))

    update_status.label = _("Обновить статус")

    def download_data(self, request, obj: Operation):
        try:
            download_yql_result_data(obj.id)
        finally:
            self.message_user(request, _("Данные для операции '%s' были успешно загружены") % str(obj))

    download_data.label = _("Загрузить данные")


# @admin.register(UserRequest)
# class UserRequestAdmin(DjangoObjectActions, admin.ModelAdmin):
#     list_display = (
#         'id', 'user', 'query', 'operation', 'format', 'status',
#         'notify_by_email', 'is_archived', 'created', 'modified',
#     )
#     list_filter = ('status', 'format', 'is_archived')
#     search_fields = (
#         'user__id',
#         'user__username',
#         'query__title',
#         'operation__yql_operation_id',
#     )
#     raw_id_fields = ('user', 'operation', 'result_data')
#     fields = (
#         'user', 'query', 'operation', 'parameters', 'format', 'status',
#         'notify_by_email', 'is_archived',
#     )
#     readonly_fields = ('operation', 'status',)


@admin.register(CourseReportRequest)
class CourseReportRequestAdmin(admin.ModelAdmin):
    list_display = (
        'id', 'course', 'user', 'query', 'format', 'status',
        'notify_by_email', 'is_archived', 'created', 'modified',
    )
    list_filter = ('status', 'format', 'is_archived')
    list_select_related = ('course', 'user', 'query')
    search_fields = (
        'course__id',
        'course__name',
        'course__code',
        'user__id',
        'user__username',
        'query__title',
        'operation__yql_operation_id',
    )
    raw_id_fields = ('course', 'user', 'result_data')
    fields = (
        'course', 'user', 'query', 'operation_link', 'result_data_link', 'parameters', 'format', 'status',
        'notify_by_email', 'notified', 'is_archived',
    )
    readonly_fields = ('operation_link', 'status', 'result_data_link')

    def operation_link(self, obj: CourseReportRequest):
        if not obj.operation_id:
            return '-'

        return format_html(
            '<a href="{}" target="_blank">{}</a>',
            reverse('admin:reports_operation_change', args=(obj.operation_id,)),
            str(obj.operation),
        )

    operation_link.short_description = _("Операция выполнения")

    def result_data_link(self, obj: CourseReportRequest):
        if not obj.result_data:
            return '-'

        return format_html(
            '<a href="{0}" target="_blank">{0}</a>',
            obj.result_data.file,
        )

    result_data_link.short_description = _("Результат по запросу")
