import json

from model_utils import FieldTracker
from model_utils.models import TimeStampedModel
from ordered_model.models import OrderedModel

from django.contrib.auth import get_user_model
from django.db import models
from django.utils.translation import ugettext_lazy as _

from kelvin.common.fields import JSONField
from kelvin.common.utils import safe_filename
from kelvin.courses.models import Course

User = get_user_model()


class QueryQuerySet(models.QuerySet):
    def active(self):
        return self.filter(is_active=True)


class Query(TimeStampedModel):
    """
    YQL-запросы
    """
    TYPE_SQLv1 = 'SQLv1'

    _TYPE_CHOICES = (
        (TYPE_SQLv1, TYPE_SQLv1),
    )

    type = models.CharField(
        _("тип запроса"),
        max_length=20,
        choices=_TYPE_CHOICES,
        default=TYPE_SQLv1,
    )

    title = models.CharField(_("название"), max_length=255)
    content = models.TextField(_("тело запроса"))
    # parameters = JSONField(
    #     _("параметры запроса"),
    #     blank=True,
    #     default={},
    # )
    version = models.IntegerField(_("версия"), blank=True, null=True)
    is_active = models.BooleanField(_("активен"), default=True)
    created_by = models.ForeignKey(
        User,
        verbose_name=_("создал"),
        related_name='queries_created_by',
        blank=True,
        null=True,
        on_delete=models.SET_NULL,
    )

    objects = QueryQuerySet.as_manager()

    class Meta:
        ordering = ('-created',)
        verbose_name = _("YQL запрос")
        verbose_name_plural = _("YQL запросы")

    def __str__(self):
        if self.version:
            return '{} [{}]'.format(self.title, self.version)
        return self.title

    @property
    def active_parameters(self):
        return self.parameters.active()


class QueryParameterQuerySet(models.QuerySet):
    def active(self):
        return self.filter(is_hidden=False)


class QueryParameter(OrderedModel):
    query = models.ForeignKey(
        Query,
        verbose_name=_("YQL запрос"),
        related_name='parameters',
        on_delete=models.CASCADE,
    )
    name = models.CharField(_("название"), max_length=255)

    TYPE_STRING = 'string'
    TYPE_NUMBER = 'number'
    TYPE_DATETIME = 'datetime'
    TYPE_BOOL = 'boolean'

    _TYPE_CHOICES = [(x, x) for x in [
        TYPE_STRING,
        TYPE_NUMBER,
        TYPE_DATETIME,
        TYPE_BOOL,
    ]]
    type = models.CharField(
        _("тип параметра"),
        max_length=20,
        choices=_TYPE_CHOICES,
        default=TYPE_STRING,
    )
    label = models.CharField(
        _("заголовок"),
        max_length=255,
    )
    help_text = models.TextField(_("описание"), blank=True)
    default = models.CharField(_("по умолчанию"), max_length=255, blank=True)
    is_hidden = models.BooleanField(_("скрытый"), default=False)

    order_with_respect_to = 'query'

    objects = QueryParameterQuerySet.as_manager()

    class Meta(OrderedModel.Meta):
        verbose_name = _("параметр запроса")
        verbose_name_plural = _("параметры запроса")

    def __str__(self):
        return '{} [{}]'.format(self.name, self.type)


class QueryOutputField(OrderedModel):
    query = models.ForeignKey(
        Query,
        verbose_name=_("YQL запрос"),
        related_name='output_fields',
        on_delete=models.CASCADE,
    )
    name = models.CharField(_("название"), max_length=255)
    title = models.CharField(_("заголовок"), max_length=255)

    order_with_respect_to = 'query'

    class Meta(OrderedModel.Meta):
        unique_together = ('query', 'name')
        verbose_name = _("заголовок поля")
        verbose_name_plural = _("заголовки полей")

    def __str__(self):
        return self.name


class OperationStatusMixin(models.Model):
    STATUS_IDLE = 'IDLE'
    STATUS_PENDING = 'PENDING'
    STATUS_RUNNING = 'RUNNING'
    STATUS_COMPLETED = 'COMPLETED'
    STATUS_ABORTING = 'ABORTING'
    STATUS_ABORTED = 'ABORTED'
    STATUS_ERROR = 'ERROR'
    STATUS_UNKNOWN = 'UNKNOWN'

    FINAL_STATUSES = (
        STATUS_COMPLETED,
        STATUS_ABORTED,
        STATUS_ERROR,
        STATUS_UNKNOWN,
    )

    _STATUS_CHOICES = [(x, x) for x in [
        STATUS_IDLE,
        STATUS_PENDING,
        STATUS_RUNNING,
        STATUS_COMPLETED,
        STATUS_ABORTING,
        STATUS_ABORTED,
        STATUS_ERROR,
        STATUS_UNKNOWN,
    ]]
    status = models.CharField(
        _("статус"),
        max_length=20,
        choices=_STATUS_CHOICES,
        default=STATUS_IDLE,
    )

    class Meta:
        abstract = True


class Operation(OperationStatusMixin, TimeStampedModel):
    """
    Операции с YQL-запросами
    """
    query = models.ForeignKey(
        Query,
        verbose_name=_("YQL запрос"),
        related_name='operations',
        on_delete=models.PROTECT,
    )

    yql_operation_id = models.CharField(_("OperationID"), max_length=255, blank=True, editable=False)

    ACTION_SAVE = 'SAVE'
    ACTION_PARSE = 'PARSE'
    ACTION_COMPILE = 'COMPILE'
    ACTION_VALIDATE = 'VALIDATE'
    ACTION_OPTIMIZE = 'OPTIMIZE'
    ACTION_RUN = 'RUN'
    ACTION_ABORT = 'ABORT'
    ACTION_STOP = 'STOP'
    ACTION_RESUME = 'RESUME'
    ACTION_EXTRACT_PARAMS_META = 'EXTRACT_PARAMS_META'

    _ACTION_CHOICES = [(x, x) for x in [
        ACTION_SAVE,
        ACTION_PARSE,
        ACTION_COMPILE,
        ACTION_VALIDATE,
        ACTION_OPTIMIZE,
        ACTION_RUN,
        ACTION_ABORT,
        ACTION_STOP,
        ACTION_RESUME,
        ACTION_EXTRACT_PARAMS_META
    ]]
    action = models.CharField(
        _("действие"),
        max_length=20,
        choices=_ACTION_CHOICES,
        default=ACTION_RUN,
    )

    """
    данные по запросу в JSON-формате:
    {
      'type': '...',
      'content': '...',
      'parameters: {...},
    }  
    """
    query_data = JSONField(_("данные запроса"))
    query_title = models.CharField(_("название запроса"), max_length=255, blank=True)
    query_version = models.IntegerField(_("версия запроса"), blank=True, null=True)

    errors = JSONField(
        _("ошибки"),
        blank=True,
        default={},
    )
    issues = JSONField(
        _("предупреждения"),
        blank=True,
        default={},
    )
    created_by = models.ForeignKey(
        User,
        verbose_name=_("создал"),
        related_name='operations_created_by',
        blank=True,
        null=True,
        on_delete=models.SET_NULL,
    )

    status_tracker = FieldTracker(fields=['status'])

    class Meta:
        ordering = ('-created',)
        verbose_name = _("операция с запросом")
        verbose_name_plural = _("операции с запросами")

    def __str__(self):
        return '{} {}'.format(
            self.action,
            self.query_title,
        )

    @property
    def operation_data(self):
        data = self.query_data

        if not self.query.parameters.exists():
            return data

        query_parameters = dict(QueryParameter.objects\
            .filter(query_id=self.query_id, default__isnull=False)\
            .values_list('name', 'default'))

        user_parameters = data.get('parameters')
        if user_parameters:
            query_parameters.update(user_parameters)

        ready_params = dict()
        for key, value in query_parameters.items():
            new_key = '${}'.format(key)
            ready_params[new_key] = json.dumps(dict(Data=str(value)))

        data['parameters'] = ready_params

        return data


class ResultFormatMixin(models.Model):
    FORMAT_CSV = 'CSV'
    # FORMAT_JSON = 'JSON'
    FORMAT_TSV = 'TSV'
    # FORMAT_TSKV = 'TSKV'
    # FORMAT_YSON_TEXT = 'YSON_TEXT'
    # FORMAT_YSON_BINARY = 'YSON_BINARY'
    FORMAT_XLSX = 'XLSX'
    # FORMAT_WIKI = 'WIKI'

    _FORMAT_CHOICES = [(x, x) for x in [
        FORMAT_XLSX,
        FORMAT_CSV,
        # FORMAT_JSON,
        FORMAT_TSV,
        # FORMAT_TSKV,
        # FORMAT_YSON_TEXT,
        # FORMAT_YSON_BINARY,
        # FORMAT_WIKI,
    ]]
    format = models.CharField(
        _("формат"),
        max_length=20,
        choices=_FORMAT_CHOICES,
    )

    class Meta:
        abstract = True

    @classmethod
    def get_format_choices(cls):
        return cls._FORMAT_CHOICES


class ResultData(ResultFormatMixin, TimeStampedModel):
    def file_upload_path(instance, filename):
        return safe_filename(filename, 'reports_results')

    operation = models.ForeignKey(
        Operation,
        verbose_name=_("операция с запросом"),
        related_name='results',
        on_delete=models.CASCADE,
    )
    old_file = models.FileField(
        _("файл"),
        upload_to=file_upload_path,
        blank=True,
        null=True,
    )
    resource = models.ForeignKey(
        to='resources.Resource',
        verbose_name=_("ресурс"),
        related_name='result_datas',
        on_delete=models.PROTECT,
    )
    size = models.PositiveIntegerField(
        _("размер файла"),
        blank=True,
        null=True,
    )
    access_count = models.IntegerField(_("кол-во скачиваний"), default=0)

    @property
    def file(self):
        return self.resource.shortened_file_url

    class Meta:
        verbose_name = _("данные по запросу")
        verbose_name_plural = _("данные по запросам")

    def __str__(self):
        return str(self.file)


class CourseReportRequestQuerySet(models.QuerySet):
    def visible(self):
        return self.filter(is_archived=False, query__is_active=True)


class CourseReportRequest(ResultFormatMixin, OperationStatusMixin, TimeStampedModel):
    course = models.ForeignKey(
        Course,
        verbose_name=_("курс"),
        related_name='report_requests',
        on_delete=models.CASCADE,
    )
    user = models.ForeignKey(
        User,
        verbose_name=_("пользователь"),
        related_name='course_report_requests',
        on_delete=models.SET_NULL,
        null=True,
        blank=True,
    )
    query = models.ForeignKey(
        Query,
        verbose_name=_("YQL запрос"),
        related_name='course_report_requests',
        on_delete=models.PROTECT,
    )
    operation = models.ForeignKey(
        Operation,
        verbose_name=_("операция выполнения"),
        related_name='course_report_requests',
        blank=True,
        null=True,
        editable=False,
        on_delete=models.PROTECT,
    )
    result_data = models.ForeignKey(
        ResultData,
        verbose_name=_("результаты по запросу"),
        related_name='course_report_requests',
        blank=True,
        null=True,
        editable=False,
        on_delete=models.PROTECT,
    )
    parameters = JSONField(
        _("параметры запроса"),
        blank=True,
        default={},
    )
    notify_by_email = models.BooleanField(
        _("отправить уведомление о выполнении на почту"),
        default=True,
    )
    notified = models.DateTimeField(
        _("уведомление отправлено"),
        null=True,
        blank=True,
    )
    is_archived = models.BooleanField(
        _("в архиве"),
        default=False,
    )

    objects = CourseReportRequestQuerySet.as_manager()

    class Meta:
        ordering = ('-created',)
        verbose_name = _("отчет по курсу")
        verbose_name_plural = _("отчеты по курсам")

    def __str__(self):
        return '{}: {}'.format(
            self.course_id, self.query,
        )

    @property
    def errors(self):
        if self.operation_id and self.status == self.STATUS_ERROR:
            return self.operation.errors

        return {}

    @property
    def issues(self):
        if self.operation_id:
            return self.operation.issues

        return {}

    @property
    def has_result_data(self):
        return bool(self.result_data_id)

    def clean(self):
        # TODO: проверить наличие параметров запроса
        parameters = self.parameters or {}

        if not self.pk:
            course_id_param = self.query.parameters.filter(name='course_id').first()
            if course_id_param:
                parameters['course_id'] = self.course_id

            self.parameters = parameters

    def save(self, *args, **kwargs):
        self.full_clean()
        super().save(*args, **kwargs)
