from django.db import transaction
from django.db.models import signals
from django.dispatch import receiver

from .models import CourseReportRequest, Operation
from .services import create_operation_by_course_report_request, update_course_report_requests_by_operation
from .signals import download_result_data_completed
from .tasks import create_yql_operation, download_yql_result_data, send_report_notification


@receiver(signal=signals.post_save, sender=CourseReportRequest)
def course_report_request_post_save_handler(instance: CourseReportRequest, created, **kwargs):
    if created and not instance.operation_id:
        transaction.on_commit(lambda: create_operation_by_course_report_request(instance.id))


@receiver(signal=signals.post_save, sender=Operation)
def operation_post_save_handler(instance: Operation, **kwargs):
    if instance.status_tracker.changed():
        update_course_report_requests_by_operation(instance.id, status=instance.status)

    if instance.status == Operation.STATUS_COMPLETED and not instance.results.exists():
        download_yql_result_data.delay(operation_id=instance.id)

    if not instance.yql_operation_id:
        transaction.on_commit(lambda: create_yql_operation.delay(operation_id=instance.id))


@receiver(signal=download_result_data_completed)
def download_result_data_completed_handler(report: CourseReportRequest, **kwargs):
    if report.notify_by_email:
        transaction.on_commit(lambda: send_report_notification(report_id=report.id))
