"""
Генерация отчетов из ORM/SQL
"""
import collections
import csv
from abc import abstractmethod
from builtins import next, object

import requests

from django.conf import settings
from django.core.files.base import ContentFile
from django.core.files.storage import default_storage
from django.db import connection

from kelvin.common.utils import safe_filename
from kelvin.reports.utils import unicode_encoder


class SqlIterator(collections.Iterator):
    """
    Итератор по результатам SQL запроса

    Возвращает tuples одинаковой длины, представляющие строчки таблицы.
    Первой строчкой может быть заголовок
    """

    def __init__(self, query, header=True):
        """
        :param query: str SQL запрос
        :param header: bool Отдавать заголовок таблицы первой строкой?
        """
        self.cursor = connection.cursor()
        self.cursor.execute(query)

        self.sent_header = not header

    def __iter__(self):
        return self

    def __next__(self):
        if not self.sent_header:
            self.sent_header = True
            return tuple(item[0] for item in self.cursor.description)

        result = self.cursor.fetchone()

        if result is None:
            raise StopIteration

        return tuple(result)


class ClickHouseIterator(collections.Iterator):
    """
    Итератор по результатам запросов в ClickHouse

    Возвращает пары (имя файла, результат), указывающие, в какой файл
    сохранить результат запроса
    """

    def __init__(self, requests, timeout=None):
        """
        :param requests: Iterable пар (имя файла, запрос), указывающие,
                         в какие файлы сохранять результаты запроса
        :param timeout: int Timeout для запроса в ClickHouse
        """
        self.requests = iter(requests)
        self.timeout = timeout

    def __iter__(self):
        return self

    def __next__(self):
        filename, query = next(self.requests)

        result = requests.post(
            settings.REPORTS_CLICKHOUSE_HOST,
            auth=(
                settings.REPORTS_CLICKHOUSE_USER,
                settings.REPORTS_CLICKHOUSE_PASSWORD
            ),
            params={'query': query},
            timeout=self.timeout,
        )

        return filename, result.text


class Reporter(object):
    """
    Создает отчеты из разных источников
    """

    @abstractmethod
    def from_iterable(self, name, iterable):
        """
        Создает отчет из iterable

        :param name: str Название отчета
        :param iterable: Iterable с результатами
        :return: list[str] Список путей к сохраненным файлов
        """
        pass


class FilesReporter(Reporter):
    """
    Сохраняет отчеты как текстовые файлы
    """

    def from_iterable(self, name, iterable):
        """
        Сохраняет отчет из iterable

        :param name: Название отчета
        :param iterable: Iterable пар (имя файла, содержание)
        :return: list[str] Список путей к сохраненным файлам
        """

        paths = []
        for filename, content in iterable:
            full_filename = safe_filename(
                '{}-{}'.format(name, filename),
                upload_dir='reports'
            )

            path = default_storage.save(full_filename, ContentFile(content))
            paths.append(path)

        return paths


class CsvReporter(Reporter):
    """
    Сохраняет отчеты в формате CSV
    """

    def __init__(self, delimiter=',', encoder=unicode_encoder):
        """
        :param delimiter: str Разделитель
        :param encoder: Функция с одним позиционным параметром, кодирующая
                        переданную строку (например в utf-8)
        """
        self.delimiter = delimiter
        self.encoder = encoder

    def _encoded_row(self, row):
        return [self.encoder(item) for item in row]

    def from_iterable(self, name, iterable):
        """
        Сохраняет CSV отчет из iterable

        :param name: Название отчета
        :param iterable: Iterable строк для CSV
        :return: list[str] Список, состоящий из одного сохраненного CSV файла
        """
        filename = safe_filename('{}.csv'.format(name), upload_dir='reports')

        with default_storage.open(filename, 'w') as stream:
            writer = csv.writer(stream, delimiter=self.delimiter)

            for row in iterable:
                writer.writerow(self._encoded_row(row))

        return [filename]
